/*
Copyright (c) 2005-2007 Lode Vandevenne
All rights reserved.

This file is part of Torenoorlog.

Torenoorlog is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Torenoorlog is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Torenoorlog.  If not, see <http://www.gnu.org/licenses/>.
*/

//Torenoorlog = Dutch word for tower wars

//copypastable compile commands:
/*
g++ *.cpp -lSDL -lGL
g++ *.cpp -lSDL -lGL -ansi -pedantic
g++ *.cpp -lSDL -lGL -Wall -Wextra -pedantic -ansi
g++ *.cpp -lSDL -lGL -Wall -Wextra -pedantic -ansi -O1
g++ *.cpp -lSDL -lGL -Wall -Wextra -pedantic -ansi -O3

./a.out
-g3
-pg
gdb
valgrind
gprof > gprof.txt
*/

#ifndef TORENOORLOG_H
#define TORENOORLOG_H

#include <SDL/SDL.h>
#include "lpi_color.h"
#include "lpi_file.h"
#include "lpi_event.h"
#include "lpi_general.h"
#include "lpi_text.h"
#include "lpi_texture.h"
#include "lpi_screen.h"
#include "lpi_gui.h"
#include "lpi_pathfind.h"
#include "lpi_math2d.h"
#include "lpi_xml.h"
#include "lpi_draw2dgl.h"
#include "lpi_audio.h"
#include "lodepng.h"

#include <vector>
#include <list>
#include <iostream>
#include <cmath>

using namespace lpi;

extern std::vector<Texture> textures;
extern std::vector<std::vector<double> > dead_sound;
extern std::vector<std::vector<double> > escape_sound;

static const size_t MAXNUMPLAYERS = 2;

class World;

//size of the textures
#define TEXSIZEX 24
#define TEXSIZEY 24
//size of the world
#define SIZEX 16
#define SIZEY 16

struct Keys
{
  unsigned long left[MAXNUMPLAYERS];
  unsigned long right[MAXNUMPLAYERS];
  unsigned long up[MAXNUMPLAYERS] ;
  unsigned long down[MAXNUMPLAYERS];
  unsigned long selectleft[MAXNUMPLAYERS];
  unsigned long selectright[MAXNUMPLAYERS];
  unsigned long exec[MAXNUMPLAYERS];
  unsigned long sell[MAXNUMPLAYERS];
  unsigned long upgrade[MAXNUMPLAYERS];
  unsigned long next[MAXNUMPLAYERS];
  
  int parse(const std::string& text, size_t& pos) //the pos parameter is given only to know where a possible error occured
  {
    xml::XMLTree tree;
    int parseresult = tree.parse(text, pos);
    if(xml::isError(parseresult)) return parseresult;
    
    for(size_t i = 0; i < tree.children.size(); i++)
    {
      xml::XMLTree* node = tree.children[i];
      if(node->attributes.empty()) continue;
      std::string type = node->attributes[0].name;
      std::string value = node->attributes[0].value;
      std::string name = node->content.name;
      
      unsigned long key;
      
      if(type == "key")
      {
        if(value.empty()) continue;
        key = value[0]; //ascii code maps directly to key
      }
      else if(type == "val") xml::unconvert(key, value);
      else continue;
      
      if(name == "left1") left[0] = key;
      else if(name == "right1") right[0] = key;
      else if(name == "up1") up[0] = key;
      else if(name == "down1") down[0] = key;
      else if(name == "selectleft1") selectleft[0] = key;
      else if(name == "selectright1") selectright[0] = key;
      else if(name == "exec1") exec[0] = key;
      else if(name == "sell1") sell[0] = key;
      else if(name == "upgrade1") upgrade[0] = key;
      else if(name == "next1") next[0] = key;
      ///////////////////////////////////////////////
      else if(name == "left2") left[1] = key;
      else if(name == "right2") right[1] = key;
      else if(name == "up2") up[1] = key;
      else if(name == "down2") down[1] = key;
      else if(name == "selectleft2") selectleft[1] = key;
      else if(name == "selectright2") selectright[1] = key;
      else if(name == "exec2") exec[1] = key;
      else if(name == "sell2") sell[1] = key;
      else if(name == "upgrade2") upgrade[1] = key;
      else if(name == "next2") next[1] = key;
    }
    
    return xml::SUCCESS;
  }
};

struct TowerType
{
  //TOWER
  int upgrades; //how much upgrades can be done
  std::vector<double> range; //range of the tower
  std::vector<double> rate; //how many shots per second
  std::vector<int> cost;
  std::vector<double> upgradetime;
  Texture* texture;
  std::string name;
  
  //TOWER&BULLET
  enum Method
  {
    NONE = 0,
    DIRECT = 1, //monster is hit directly, bullet doesn't do anything anymore except being drawn
    SPLASHMISSILE = 2, //flies to target and explodes there
    SPLASHRANGE = 3, //splash directly around the tower, this creates no bullet, just a translucent circle around the tower
    BOOST = 4 //boosts damage of towers around it
  } method;
  
  enum Effect
  {
    DAMAGE = 1,
    SLOW = 2
  } effect;
  
  std::vector<double> damage;
  
  //BULLET
  Texture* bullet_texture;
  double bullet_effect_speed; //seconds to go from start to finish
  std::vector<double> splash; //radius of splash damage
  
  
  void parseXML(const std::string& in, xml::ElementPos& pos)
  {
    xml::ElementPos elementpos;
    std::string elementname;
    
    while(parseElement(in, pos.cb, pos.ce, elementname, elementpos) == xml::SUCCESS)
    {
      if(elementname == "method")
      {
        unsigned long m;
        xml::unconvert(m, in, elementpos.cb, elementpos.ce);
        method = Method(m);
        switch(method)
        {
          case DIRECT: bullet_effect_speed = 0.1; break;
          case SPLASHMISSILE: bullet_effect_speed = 0.5; break;
          default: bullet_effect_speed = 0.1; break;
        }
      }
      else if(elementname == "effect")
      {
        unsigned long m;
        xml::unconvert(m, in, elementpos.cb, elementpos.ce);
        effect = Effect(m);
      }
      else if(elementname == "range")
      {
        range.resize(range.size() + 1);
        xml::unconvert(range.back(), in, elementpos.cb, elementpos.ce);
      }
      else if(elementname == "rate")
      {
        rate.resize(rate.size() + 1);
        xml::unconvert(rate.back(), in, elementpos.cb, elementpos.ce);
      }
      else if(elementname == "cost")
      {
        cost.resize(cost.size() + 1);
        xml::unconvert(cost.back(), in, elementpos.cb, elementpos.ce);
      }
      else if(elementname == "splash")
      {
        splash.resize(splash.size() + 1);
        xml::unconvert(splash.back(), in, elementpos.cb, elementpos.ce);
      }
      else if(elementname == "damage")
      {
        damage.resize(damage.size() + 1);
        xml::unconvert(damage.back(), in, elementpos.cb, elementpos.ce);
      }
      else if(elementname == "texture")
      {
        int index;
        xml::unconvert(index, in, elementpos.cb, elementpos.ce);
        texture = &textures[index];
      }
      else if(elementname == "bullet_texture")
      {
        int index;
        xml::unconvert(index, in, elementpos.cb, elementpos.ce);
        bullet_texture = &textures[index];
      }
      else if(elementname == "upgrades") xml::unconvert(upgrades, in, elementpos.cb, elementpos.ce);
      else if(elementname == "upgradetime")
      {
        upgradetime.resize(upgradetime.size() + 1);
        xml::unconvert(upgradetime.back(), in, elementpos.cb, elementpos.ce);
      }
      else if(elementname == "name") xml::unconvert(name, in, elementpos.cb, elementpos.ce);
    }
  }
  
  int getCost(size_t level = 0) const
  {
    if(cost.size() > level) return cost[level];
    else if(cost.empty()) return 0;
    else return cost.back();
  }
  
  int getFullCost(size_t level) const //get the full cost of all upgrades up to this level
  {
    int result = 0;
    for(size_t i = 0; i <= level && i < cost.size(); i++) result += cost[i];
    return result;
  }
  
  double getRange(size_t level = 0) const
  {
    if(range.size() > level) return range[level];
    else if(range.empty()) return 0;
    else return range.back();
  }
  
  double getDamage(size_t level = 0) const
  {
    if(damage.size() > level) return damage[level];
    else if(damage.empty()) return 0;
    else return damage.back();
  }
  
  double getRate(size_t level = 0) const
  {
    if(rate.size() > level) return rate[level];
    else if(rate.empty()) return 0;
    else return rate.back();
  }
  
  double getSplash(size_t level = 0) const
  {
    if(splash.size() > level) return splash[level];
    else if(splash.empty()) return 0;
    else return splash.back();
  }
  
  double getUpgradeTime(size_t level = 0) const
  {
    if(upgradetime.size() > level) return upgradetime[level];
    else if(upgradetime.empty()) return 0;
    else return upgradetime.back();
  }
  
};

struct MonsterType
{
  double speed; //tiles per second
  Texture* texture;
  double basehealth;
  double basegold;
  double basecost; //cost to be sent by player
  unsigned long wavesize; //with how many they come in a wave
  int chance; //chance to appear, set to 0 for not having it appear by computer (but by player factories)
  bool user; //if true, can be sent by user to other player using the Selector
  std::string name; //currently only used for user-sendable monsters
  
  bool immune; //if true: then they won't be slowed down by slow towers
  
  double getUserWaveCost(int level, double lin_gold, double sq_gold, double exp_gold) const
  {
    double linear = level * lin_gold * basecost;
    double quadratic = (level * level) * sq_gold * basecost;
    double exponential = basecost * std::pow(exp_gold, level);
    
    double cost = linear + quadratic + exponential;
    return cost * wavesize;
  }
  
  
  double getHealth(int level, double lin_health, double sq_health, double exp_health, double tweak_health, double difficulty_modifier) const
  {
    double linear = level * lin_health * basehealth;
    double quadratic = (level * level) * sq_health * basehealth;
    double exponential = basehealth * std::pow(exp_health, level);
    
    double health = linear + quadratic + exponential;
    return tweak_health * health * difficulty_modifier;
  }
  
  double getGold(int level, double lin_gold, double sq_gold, double exp_gold, double tweak_gold) const
  {
    double linear = level * lin_gold * basegold;
    double quadratic = (level * level) * sq_gold * basegold;
    double exponential = basegold * std::pow(exp_gold, level);
    
    return tweak_gold * (linear + quadratic + exponential);
  }
  
  void parseXML(const std::string& in, xml::ElementPos& pos)
  {
    xml::ElementPos elementpos;
    std::string elementname;
    
    while(parseElement(in, pos.cb, pos.ce, elementname, elementpos) == xml::SUCCESS)
    {
      if(elementname == "speed") xml::unconvert(speed, in, elementpos.cb, elementpos.ce);
      else if(elementname == "basehealth") xml::unconvert(basehealth, in, elementpos.cb, elementpos.ce);
      else if(elementname == "basegold") xml::unconvert(basegold, in, elementpos.cb, elementpos.ce);
      else if(elementname == "basecost") xml::unconvert(basecost, in, elementpos.cb, elementpos.ce);
      else if(elementname == "wavesize") xml::unconvert(wavesize, in, elementpos.cb, elementpos.ce);
      else if(elementname == "chance") xml::unconvert(chance, in, elementpos.cb, elementpos.ce);
      else if(elementname == "immune") xml::unconvert(immune, in, elementpos.cb, elementpos.ce);
      else if(elementname == "user") xml::unconvert(user, in, elementpos.cb, elementpos.ce);
      else if(elementname == "name") xml::unconvert(name, in, elementpos.cb, elementpos.ce);
      else if(elementname == "texture")
      {
        int index;
        xml::unconvert(index, in, elementpos.cb, elementpos.ce);
        texture = &textures[index];
      }
    }
  }
  
  MonsterType()
  {
    user = false;
  }
};

//drawBar: for drawing health and progress bars
inline void drawBar(int x, int y, const ColorRGB& color1, const ColorRGB color2, double current, double maximum)
{
  const int w = 20; //bar width
  const int h = 3; //bar height
  int c = int(w * current / maximum);
  drawRectangle(x, y, x + c, y + h, color1);
  drawRectangle(x + c, y, x + w, y + h, color2);
}

class Monster
{
  public:
  bool deleted; //true if this monster is dead or escaped --> indication that the world must delete it
  bool dead; //only if dead it'll give gold
  bool escaped; //only if escaped the monster will make you lose a life
  
  Vector2 pos; //in tile coordinates
  
  const MonsterType* type;
  
  double starthealth; //how much health this monster started with, used for drawing the health bar
  double health;
  
  std::vector<PathFindPos> path;
  std::vector<PathFindPos> testpath; //this allows not recalculating twice when testing if new path will work
  size_t path_index; //used for knowing where on the path the tile on which the monster is now, is
  
  World* world; //the owner of this monster is the world
  
  int level;
  
  double gold; //how much gold this particular monster will give when it dies
  
  double slowed; //how much this monster is slowed by towers
  double slowTime;
  
  double getSpeed()
  {
    return (1.0 - slowed) * type->speed;
  }
  
  Monster(World* world, double gold)
  {
    this->world = world;
    path_index = 0;
    deleted = false;
    dead = false;
    escaped = false;
    this->gold = gold;
    slowed = 0.0;
  }
  
  void slow(double amount, double time) //time is the CURRENT time, not the time how long to slow. amount is how much the speed is slowed.
  {
    if(type->immune) return;
    if(slowed <= amount)
    {
      slowed = amount;
      if(slowed > 0.8) slowed = 0.8;
      slowTime = time;
    }
  }
  
  bool slowable() const
  {
    if(type->immune) return false;
    if(slowed) return false;
    return true;
  }
  
  void hurt(double damage)
  {
    health -= damage;
    if(!deleted && health <= 0.0) deleted = dead = true;
  }
  
  void hit(const TowerType* t, int level, double time)
  {
    if(t->effect == TowerType::DAMAGE)
    {
      hurt(t->getDamage(level));
    }
    else if(t->effect == TowerType::SLOW)
    {
      slow(t->getDamage(level), time);
    }
  }
  
  void draw(int drawx, int drawy)
  {
    if(deleted) return; //to avoid weird health bars being drawn
    
    type->texture->drawCentered(drawx + int(TEXSIZEX * pos.x), drawy + int(TEXSIZEY * pos.y));
    if(slowed > 0) textures[13].drawCentered(drawx + int(TEXSIZEX * pos.x), drawy + int(TEXSIZEY * pos.y));
    
    int x = drawx - TEXSIZEX / 2 + int(TEXSIZEX * pos.x);
    int y = drawy - TEXSIZEY / 2 + int(TEXSIZEY * pos.y);
    drawBar(x, y, ColorRGB(128, 255, 128, 192), ColorRGB(255, 128, 128, 192), health, starthealth);
  }
  
  bool calculatePath(); //stores result in testpath, use useNewPath to finalize
  
  void useNewPath() //use the testpath as path (by swapping the vectors)
  {
    path_index = 0;
    path.swap(testpath);
  }
  
  void handle(double dt, double time);
};

class Tower
{
  public:
  Vector2 pos;
  bool firstshot; //to get the timing correct
  size_t level;
  
  TowerType* type;
  
  double lastTime; //last time when shot
  
  World* world; //the owner of this monster is the world
  
  bool upgrading;
  double upgradeStartTime;
  double upgradeEndTime;
  double upgradePercentage; //used for drawing the bar
  
  void startUpgrade(double time, bool started)
  {
    //level++;
    upgrading = true;
    upgradeStartTime = time;
    if(!started) upgradeEndTime = upgradeStartTime; //if the game didn't start yet, upgrade immediatly
    else upgradeEndTime = time + type->getUpgradeTime(level);
  }
  
  Tower(World* world, TowerType* type)
  {
    this->world = world;
    this->type = type;
    firstshot = false;
    level = 0;
    upgrading = 0;
  }
  
  int getPrintedLevel() { return level + 1; }
  
  void draw(int drawx, int drawy)
  {
    int x = drawx + int(TEXSIZEX * (pos.x - 0.5));
    int y = drawy + int(TEXSIZEY * (pos.y - 0.5));
    if(isFullyUpgraded()) textures[12].draw(x, y);
    else textures[11].draw(x, y);
    type->texture->draw(x, y);
    if(level > 0)
    {
      print(getPrintedLevel(), x + 4, y + TEXSIZEY - 10, Markup(RGB_White, ColorRGB(0, 0, 0, 255), 1, &builtInFont8x8));
    }
    
    if(upgrading)
    {
      drawBar(x, y + TEXSIZEY - 8, ColorRGB(128, 128, 255, 255), ColorRGB(255, 255, 128, 255), upgradePercentage, 100);
    }
  }
  
  void handle(double dt, double time);
  
  int getCost() const
  {
    return type->getCost(level);
  }
  
  
  bool isFullyUpgraded() const
  {
    return int(level) >= type->upgrades;
  }
  
  int getUpgradePrice() const
  {
    return type->getCost(level + 1);
  }
  
  double getRange() const
  {
    return type->getRange(level);
  }
  
  double getDamage() const
  {
    return type->getDamage(level);
  }
};

class Bullet
{
  public:
  
  TowerType* type;
  World* world; //the parent of this bullet
  
  Vector2 start;
  Vector2 finish;
  double current;
  bool deleted;
  size_t level;
  
  Bullet(int level)
  {
    deleted = false;
    current = 0.0;
    this->level = level;
  }
  
  void handle(double dt, double /*time*/);
  
  void draw(int drawx, int drawy)
  {
    Vector2 c = start + current * (finish - start);
    
    type->bullet_texture->drawCentered(drawx + (int)(TEXSIZEX * c.x), drawy + (int)(TEXSIZEY * c.y));
  }
};

class Selector : public gui::Element
{
  public:

  enum OptionType //I use it only for building towers now, sell and upgrade have separate key
  {
    NOTHING,
    TOWER,
    MONSTER,
    SELL,
    UPGRADE
  };
  
  struct Option
  {
    TowerType* tower;
    MonsterType* monster;
    OptionType type;
    
    Option()
    {
      tower = 0;
      monster = 0;
      type = NOTHING;
    }
  };
  
  std::vector<Option> options;
  
  int sel;
  
  bool monsters_ready; //set by game, determines if monsters are drawn as available or not
  
  void setCorrectSize()
  {
    setSizex(options.size() * TEXSIZEX);
    setSizey(TEXSIZEY);
  }
  
  void addTower(TowerType* tower)
  {
    options.resize(options.size() + 1);
    options.back().type = TOWER;
    options.back().tower = tower;
    setCorrectSize();
  }
  
  void addMonster(MonsterType* monster)
  {
    options.resize(options.size() + 1);
    options.back().type = MONSTER;
    options.back().monster = monster;
    setCorrectSize();
  }
  
  Selector()
  {
    sel = 0;
  }
  
  ~Selector()
  {
  }
  
  void wrapController()
  {
    if(sel < 0) sel = options.size() - 1;
    if(sel >= int(options.size())) sel = 0;
  }
  
  OptionType getOptionType() const
  {
    if(sel >= 0 && sel < int(options.size())) return options[sel].type;
    else return NOTHING;
  }
  
  TowerType* getOptionTower() const
  {
    if(sel >= 0 && sel < int(options.size())) return options[sel].tower;
    else return 0;
  }
  
  MonsterType* getOptionMonster() const
  {
    if(sel >= 0 && sel < int(options.size())) return options[sel].monster;
    else return 0;
  }
  
  virtual void handleWidget()
  {
    if(pressed())
    {
      int x = mouseGetRelPosX() / TEXSIZEX;
      sel = x;
    }
  }
  
  virtual void drawWidget() const
  {
    for(size_t i = 0; i < options.size(); i++)
    {
      Texture* tx;
      ColorRGB colorMod = RGB_White;
      switch(options[i].type)
      {
        case TOWER: tx = options[i].tower->texture; break;
        case MONSTER:
          tx = options[i].monster->texture;
          if(!monsters_ready) colorMod = ColorRGB(255, 255, 255, 128);
          break;
        default: tx = &textures[0]; break;
      }
      tx->draw(x0 + TEXSIZEX * i, y0, colorMod);
    }
    
    textures[10].draw(x0 + TEXSIZEX * sel, y0);
  }
};

static const double TIMEBETWEENMONSTERS = 0.2;

struct WaveCreator //a wave of monsters
{
  std::vector<Monster*> to_spawn;
  double lastMainSpawnTime; //last time an auto-wave was created
  double lastSingleSpawnTime; //last time a single monster appeared in the world
  std::vector<const MonsterType*> waveorder; //the order in which waves will appear
  size_t current_spawn;
  int level;
  double last_health; //only used by info display
  
  //increasing stats for monsters
  double lin_health; //linear health increase
  double sq_health; //quadratic health increase
  double exp_health; //exponential health increase. Current HP will be: level*lin + base*(exp^level), where level starts at 0
  double tweak_health;
  double lin_gold; //idem for gold
  double sq_gold; //quadratic gold increase
  double exp_gold; //idem for gold
  double tweak_gold;
  
  void reset()
  {
    lastMainSpawnTime = -99999.9;
    lastSingleSpawnTime = -99999.9;
    current_spawn = 0;
    level = 0;
    last_health = 0;
    to_spawn.clear();
  }
  
  bool inPipeline() const
  {
    return !to_spawn.empty();
  }
  
  double timeTillNextWave(double time, double time_between_waves) const //if it returns <= 0, it's time for the next wave
  {
    return time_between_waves + lastMainSpawnTime - time;
  }
  
  WaveCreator()
  {
    reset();
  }
  
  void createSpawn(World* world, const MonsterType* type, double health_modifier) //used for the auto-spawns, and called externally for extra player-sent spawns
  {
    size_t amount = type->wavesize;
    for(size_t i = 0; i < amount; i++)
    {
      Monster* m = new Monster(world, type->getGold(level, lin_gold, sq_gold, exp_gold, tweak_gold));
      m->type = type;
      m->starthealth = m->health = type->getHealth(level, lin_health, sq_health, exp_health, tweak_health, health_modifier);
      m->level = level;
      to_spawn.push_back(m);
    }
  }
  
  void spawnNow()
  {
    lastMainSpawnTime = -99999.0;
  }
  
  void createSpawn(World* world, double health_modifier)
  {
    if(waveorder.empty()) return;
    
    createSpawn(world, waveorder[current_spawn], health_modifier);
    
    current_spawn++;
    if(current_spawn >= waveorder.size()) current_spawn = 0;
  }
  
  void handle(double dt, double time, World* world, double health_modifier, double time_between_waves);
  
  void createWaveOrder(const std::vector<MonsterType>& monstertypes)
  {
    waveorder.clear();
    if(monstertypes.empty()) return;
    size_t sum = 0;
    std::vector<size_t> chances;
    for(size_t i = 0; i < monstertypes.size(); i++)
    {
      chances.push_back(monstertypes[i].chance);
      sum += monstertypes[i].chance;
    }
    
    while(sum > 1000) //avoid too large vector...
    {
      for(size_t i = 0; i < chances.size(); i++)
      {
        chances[i] /= 2;
      }
    }
    
    for(size_t i = 0; i < chances.size(); i++)
    for(size_t j = 0; j < chances[i]; j++)
    {
      waveorder.push_back(&monstertypes[i]);
    }
    
    //now reorder the order of waveorder
    unsigned long twiddle = 1;
    for(size_t i = 0; i < waveorder.size(); i++)
    {
      twiddle = (1664525 * twiddle + 1013904223) % 4294967295U;
      size_t amount = waveorder.size() - i;
      size_t index = i + (twiddle % amount);
      std::swap(waveorder[i], waveorder[index]);
    }
    
  }
};

class Message
{
  public:
  double lastTime;
  
  void set(const std::string& text)
  {
    this->text = text;
    lastTime = getSeconds();
  }
  
  void draw(int x, int y, double time)
  {
    const double duration = 4.0;
    if(time - lastTime < duration)
    {
      ColorRGB color1(255, 255, 255, 255);
      ColorRGB color2(255, 0, 0, 0);
      double a = (time - lastTime) / duration;
      ColorRGB color = color2 & a | color1 & (1.0 - a);
      printCentered(text, x, y, Markup(color, ColorRGB(0, 0, 0, 255 - int(a * 255)), 1));
    }
  }
  
  private:
  std::string text;
};

struct Player
{
  double gold;
  int lives;
  double score;
  
  double lastUserWaveTime; //last time you sent a wave to the opponent
  
  Selector selector;
  Message message;
  
  bool canSpawnUserWave(double time, double time_between_waves) const
  {
    return time - lastUserWaveTime > time_between_waves;
  }
  
  void reset(double start_gold, int start_lives)
  {
    score = 0;
    gold = start_gold;
    lives = start_lives;
    lastUserWaveTime = -99999.0;
  }
};

class World : public World_AStar, public gui::Element //there is one Game, but one or two Worlds: two, if there are two players.
{
  public:
  int cmap[SIZEX][SIZEY]; //collsion values. 0 = free, 1 = tower is placed here, 2 = place where monsters start, 3 = end
  int tmap[SIZEX][SIZEY]; //texture map. 0=a, 1=b, 2=start, 3=end
  int controlx, controly;
  
  int getsizex() const { return SIZEX; }
  int getsizey() const { return SIZEY; }
  
  Player player;
  WaveCreator waves;
  
  std::vector<PathFindPos> testpath; //used when trying if path for future monsters will work
  
  std::list<Monster*> monsters;
  std::list<Tower*> towers;
  std::list<Bullet*> bullets;
  
  int startfieldx0;
  int startfieldy0;
  int startfieldx1;
  int startfieldy1;
  
  int endx;
  int endy;
  
  int wants_to_next_level;
  
  bool playerWantsToNextLevel() const
  {
    if(wants_to_next_level >= waves.level) return true;
    else return false;
  }
  
  void makemaps()
  {
    for(size_t y = 0; y < SIZEX; y++)
    for(size_t x = 0; x < SIZEX; x++)
    {
      cmap[x][y] = 0;
      if((x+1) / 2 == SIZEX / 4 && (y+1) / 2 == SIZEY / 4) tmap[x][y] = 1;
      else tmap[x][y] = 0;
    }
    
    for(int y = startfieldy0; y < startfieldy1; y++)
    for(int x = startfieldx0; x < startfieldx1; x++)
    {
      tmap[x][y] = 2;
      cmap[x][y] = 2;
    }
    
    tmap[endx][endy] = 3;
    cmap[endx][endy] = 3;
  }
  
  void reset()
  {
    //player.reset();
    monsters.clear();
    bullets.clear();
    towers.clear();
    
    makemaps();
    
    waves.reset();
    
    wants_to_next_level = -1;
  }
  
  bool isInside(const Vector2& pos)
  {
    return pos.x >= 0 && pos.y >= 0 && pos.x <= double(getsizex()) && pos.y <= double(getsizey());
  }
  
  Vector2 getRandomStartPos()
  {
    Vector2 result;
    result.x = getRandom((double)startfieldx0, (double)startfieldx1);
    result.y = getRandom((double)startfieldy0, (double)startfieldy1);
    return result;
  }
  
  Monster* getMonsterNear(const Vector2& pos, double radius, bool slowable) //used by towers to find nearby monster to shoot
  {
    for(std::list<Monster*>::iterator it = monsters.begin(); it != monsters.end(); ++it)
    {
      if(slowable && !(*it)->slowable()) continue;
      double dist = distance((*it)->pos, pos);
      if(dist <= radius) return *it;
    }
    return 0;
  }
  
  void splashDamage(const Vector2& pos, TowerType* t, int level, double time)
  {
    double radius = t->getSplash(level);
    for(std::list<Monster*>::iterator it = monsters.begin(); it != monsters.end(); ++it)
    {
      Monster* m = *it;
      if(distance(m->pos, pos) <= radius)
      {
        m->hit(t, level, time);
      }
    }
  }
  
  void wrapController()
  {
    if(controlx < 0) controlx = SIZEX - 1;
    if(controly < 0) controly = SIZEY - 1;
    if(controlx >= SIZEX) controlx = 0;
    if(controly >= SIZEY) controly = 0;
  }
  
  bool calculateAllPaths(); //doesn't store them in the paths yet, use useNewPaths after this
  bool testTowerPlacement(int x, int y); //if it returns true, the paths are all calculated and you can make the world and monsters use it with useNewPaths(), at least if you set the tile x,y to blocking then
  
  bool testTowerPlacement()
  {
    return testTowerPlacement(controlx, controly);
  }
  
  void useNewPaths()
  {
    for(std::list<Monster*>::iterator it = monsters.begin(); it != monsters.end(); ++it)
    {
      (*it)->useNewPath();
    }
  }
  
  void setStartAndEnd(int startfieldx0, int startfieldy0, int startfieldx1, int startfieldy1, int endx, int endy)
  {
    
    this->startfieldx0 = startfieldx0;
    this->startfieldy0 = startfieldy0;
    this->startfieldx1 = startfieldx1;
    this->startfieldy1 = startfieldy1;
    this->endx = endx;
    this->endy = endy;
    
    makemaps();
    
    controlx = startfieldx0;
    controly = startfieldy0;
  }
  
  World()
  {
    setSizex(SIZEX * TEXSIZEX);
    setSizey(SIZEY * TEXSIZEY);
  }
  
  ~World()
  {
    for(std::list<Monster*>::iterator it = monsters.begin(); it != monsters.end(); ++it)
    {
      delete *it;
    }
    
    for(std::list<Tower*>::iterator it = towers.begin(); it != towers.end(); ++it)
    {
      delete *it;
    }
    
    for(std::list<Bullet*>::iterator it = bullets.begin(); it != bullets.end(); ++it)
    {
      delete *it;
    }
  }
  
  virtual bool blocks(int x, int y) const
  {
    if(x < 0 || y < 0 || x >= getsizex() || y >= getsizey()) return true;
    return cmap[x][y] == 1;
  }
  
  bool towerplacable(int x, int y) const
  {
    if(x < 0 || y < 0 || x >= getsizex() || y >= getsizey()) return false;
    return cmap[x][y] == 0;
  }
  
  bool towerplacable() const
  {
    return towerplacable(controlx, controly);
  }
  
  virtual void handleWidget()
  {
    if(pressed())
    {
      controlx = mouseGetRelPosX() / TEXSIZEX;
      controly = mouseGetRelPosY() / TEXSIZEY;
    }
  }
  
  virtual void drawWidget() const
  {
    for(size_t y = 0; y < SIZEX; y++)
    for(size_t x = 0; x < SIZEX; x++)
    {
      if(tmap[x][y] == 0) textures[0].draw(x0 + TEXSIZEX * x, y0 + TEXSIZEY * y);
      else if(tmap[x][y] == 1) textures[1].draw(x0 + TEXSIZEX * x, y0 + TEXSIZEY * y);
      else if(tmap[x][y] == 2) textures[2].draw(x0 + TEXSIZEX * x, y0 + TEXSIZEY * y);
      else if(tmap[x][y] == 3) textures[3].draw(x0 + TEXSIZEX * x, y0 + TEXSIZEY * y);
    }
    
    for(std::list<Monster*>::const_iterator it = monsters.begin(); it != monsters.end(); ++it)
    {
      (*it)->draw(x0, y0);
    }
    
    for(std::list<Tower*>::const_iterator it = towers.begin(); it != towers.end(); ++it)
    {
      (*it)->draw(x0, y0);
    }
    
    for(std::list<Bullet*>::const_iterator it = bullets.begin(); it != bullets.end(); ++it)
    {
      (*it)->draw(x0, y0);
    }
  
    textures[10].draw(x0 + TEXSIZEX * controlx, y0 + TEXSIZEY * controly);
  }
  
  void deleteTowerAt(int x, int y)
  {
    for(std::list<Tower*>::iterator it = towers.begin(); it != towers.end(); )
    {
      Tower* t = *it;
      if(int(t->pos.x) == x && int(t->pos.y) == y)
      {
        delete t;
        it = towers.erase(it);
      }
      else ++it;
    }
  }
  
  void deleteTowerAt()
  {
    deleteTowerAt(controlx, controly);
  }
  
  Tower* getTowerAt(int x, int y)
  {
    for(std::list<Tower*>::iterator it = towers.begin(); it != towers.end(); ++it)
    {
      Tower* t = *it;
      if(int(t->pos.x) == x && int(t->pos.y) == y)
      {
        return t;
      }
    }
    return 0;
  }
  
  Tower* getTowerAt()
  {
    return getTowerAt(controlx, controly);
  }
  
  void removeDeletedObjects()
  {
    for(std::list<Monster*>::iterator it = monsters.begin(); it != monsters.end(); )
    {
      if((*it)->deleted)
      {
        
        if((*it)->dead)
        {
          player.gold += (*it)->gold;
          player.score += (*it)->gold;
          audioPlay(dead_sound[0]);
        }
        if((*it)->escaped)
        {
          player.lives--;
          audioPlay(escape_sound[0]);
        }
        delete *it;
        it = monsters.erase(it);
      }
      else ++it;
    }
    
    for(std::list<Bullet*>::iterator it = bullets.begin(); it != bullets.end(); )
    {
      if((*it)->deleted)
      {
        delete *it;
        it = bullets.erase(it);
      }
      else ++it;
    }
  }
  
  void handle(double dt, double time) //dt = delta time
  {
    gui::Element::handle();
    
    removeDeletedObjects();
    
    for(std::list<Monster*>::iterator it = monsters.begin(); it != monsters.end(); ++it)
    {
      (*it)->handle(dt, time);
    }
    
    for(std::list<Tower*>::iterator it = towers.begin(); it != towers.end(); ++it)
    {
      (*it)->handle(dt, time);
    }
    
    for(std::list<Bullet*>::iterator it = bullets.begin(); it != bullets.end(); ++it)
    {
      (*it)->handle(dt, time);
    }
    
  }
  
  void addBullet(Vector2 start, Vector2 finish, Tower* tower)
  {
    Bullet* b = new Bullet(tower->level);
    b->type = tower->type;
    b->start = start;
    b->finish = finish;
    b->level = tower->level;
    bullets.push_back(b);
    b->world = this;
  }
};

struct Game
{
  double start_gold;
  int start_lives;
  
  double time_between_waves;
  
  
  World world[MAXNUMPLAYERS];
  
  enum
  {
    PREGAME, //not started yet --> game board not drawn, main menu with game options is shown
    PLAYING0, //the buildup phase, no activity other than building by players
    PLAYING1, //game started, monsters come, towers shoot, ...
    GAMEOVER //game board drawn but paused and gameover window visible
  } state;
  
  enum
  {
    SANDBOX, //like normal, but you start with 10 times more lives and 100 times more gold
    EASY, //monsters have 1.5x less hitpoints
    NORMAL,
    HARD, //monsters have 1.5x more hitpoints
    NIGHTMARE //monsters have 2x more hitpoints
  } difficulty;
  
  enum
  {
    SINGLEPLAYER, //one player, the other board isn't drawn
    TWOPLAYER, //two players competing against each other
    COOP //not implemented
  } mode;
  
  bool paused;
  
  std::vector<MonsterType> monstertypes;
  std::vector<TowerType> towertypes;
  
  Keys keys;
  
  //cost applied to selling a tower to avoid juggling
  int juggle_cost_base;
  double juggle_cost_lin;
  double juggle_cost_sq;
  double juggle_cost_exp;
  
  double sell_factor; //selling price is this * full price
  
  std::string mod_author;
  
  Game()
  {
    paused = false;
    state = PREGAME;
    mod_author = "standard game";
  }
  
  double getHealthModifier() //modifier for monster health,depends on difficulty
  {
    switch(difficulty)
    {
      case SANDBOX: return 1.0;
      case EASY: return 0.66;
      case NORMAL: return 1.0;
      case HARD: return 1.5;
      case NIGHTMARE: return 2.0;
    }
    
    return 1.0;
  }
  
  double getStartGoldModifier() //depends on difficulty
  {
    if(difficulty == SANDBOX) return 100.0;
    else return 1.0;
  }
  
  double getStartLivesModifier() //depends on difficulty
  {
    if(difficulty == SANDBOX) return 10.0;
    else return 1.0;
  }
  
  size_t getNumPlayers() const
  {
    if(mode == SINGLEPLAYER) return 1;
    else if(mode == TWOPLAYER) return 2;
    return 0;
  }
  
  bool active() const //returns false if gameover or not started
  {
    return (state == PLAYING1) && !paused;
  }
  
  void reset() //this allows restarting the game but does NOT reset parsed content. For a more full reset, do parse() again
  {
    for(size_t i = 0; i < MAXNUMPLAYERS; i++)
    {
      world[i].reset();
      world[i].player.reset(getStartGoldModifier() * start_gold, int(getStartLivesModifier() * start_lives));
    }
  }
  
  double getJugglingPenalty(int level) const //level is the wave level
  {
    double linear = level * juggle_cost_lin;
    double quadratic = (level * level) * juggle_cost_sq;
    double exponential = juggle_cost_base * std::pow(juggle_cost_exp, level);
    return linear + quadratic + exponential;
  }
  
  double getJugglingPenalty() const
  {
    return getJugglingPenalty(world[0].waves.level);
  }
  
  double getSellingPrice(const Tower* t, int level, bool started) //can be negative due to juggling penalty; level is the wave level, not tower level
  {
    double price = t->type->getFullCost(t->level);
    if(!started) return price; //if the game didn't start yet, sell without penalty
    price *= sell_factor;
    price -= getJugglingPenalty(level);
    return price;
  }
  
  bool bothPlayersWantToNextLevel() const
  {
    if(getNumPlayers() == 1) return world[0].playerWantsToNextLevel();
    else return world[0].playerWantsToNextLevel() && world[1].playerWantsToNextLevel();
  }
  
  void parseXML(const std::string& in, xml::ElementPos& pos)
  {
    monstertypes.clear();
    
    xml::ElementPos elementpos;
    std::string elementname;
    
    WaveCreator waves_prototype;
    
    while(parseElement(in, pos.cb, pos.ce, elementname, elementpos) == xml::SUCCESS)
    {
      if(elementname == "monstertype")
      {
        monstertypes.resize(monstertypes.size() + 1);
        monstertypes.back().parseXML(in, elementpos);
      }
      else if(elementname == "towertype")
      {
        towertypes.resize(towertypes.size() + 1);
        towertypes.back().parseXML(in, elementpos);
      }
      else if(elementname == "lin_health") xml::unconvert(waves_prototype.lin_health, in, elementpos.cb, elementpos.ce);
      else if(elementname == "sq_health") xml::unconvert(waves_prototype.sq_health, in, elementpos.cb, elementpos.ce);
      else if(elementname == "exp_health") xml::unconvert(waves_prototype.exp_health, in, elementpos.cb, elementpos.ce);
      else if(elementname == "tweak_health") xml::unconvert(waves_prototype.tweak_health, in, elementpos.cb, elementpos.ce);
      else if(elementname == "lin_gold") xml::unconvert(waves_prototype.lin_gold, in, elementpos.cb, elementpos.ce);
      else if(elementname == "sq_gold") xml::unconvert(waves_prototype.sq_gold, in, elementpos.cb, elementpos.ce);
      else if(elementname == "exp_gold") xml::unconvert(waves_prototype.exp_gold, in, elementpos.cb, elementpos.ce);
      else if(elementname == "tweak_gold") xml::unconvert(waves_prototype.tweak_gold, in, elementpos.cb, elementpos.ce);
      else if(elementname == "start_gold") xml::unconvert(start_gold, in, elementpos.cb, elementpos.ce);
      else if(elementname == "start_lives") xml::unconvert(start_lives, in, elementpos.cb, elementpos.ce);
      else if(elementname == "juggle_cost_base") xml::unconvert(juggle_cost_base, in, elementpos.cb, elementpos.ce);
      else if(elementname == "juggle_cost_lin") xml::unconvert(juggle_cost_lin, in, elementpos.cb, elementpos.ce);
      else if(elementname == "juggle_cost_sq") xml::unconvert(juggle_cost_sq, in, elementpos.cb, elementpos.ce);
      else if(elementname == "juggle_cost_exp") xml::unconvert(juggle_cost_exp, in, elementpos.cb, elementpos.ce);
      else if(elementname == "sell_factor") xml::unconvert(sell_factor, in, elementpos.cb, elementpos.ce);
      else if(elementname == "mod_author") xml::unconvert(mod_author, in, elementpos.cb, elementpos.ce);
      else if(elementname == "time_between_waves") xml::unconvert(time_between_waves, in, elementpos.cb, elementpos.ce);
    }
    
    for(size_t j = 0; j < MAXNUMPLAYERS; j++)
    {
      for(size_t i = 0; i < towertypes.size(); i++)
      {
        world[j].player.selector.addTower(&towertypes[i]);
      }
      
      for(size_t i = 0; i < monstertypes.size(); i++)
      {
        if(monstertypes[i].user)
          world[j].player.selector.addMonster(&monstertypes[i]);
      }
      
      
      world[j].waves = waves_prototype;
    }
  }
};



#endif
