/*
Copyright (c) 2005-2007 Lode Vandevenne
All rights reserved.

This file is part of Torenoorlog.

Torenoorlog is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Torenoorlog is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Torenoorlog.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "torenoorlog.h"

#include <SDL/SDL.h>
#include "lpi_color.h"
#include "lpi_file.h"
#include "lpi_event.h"
#include "lpi_general.h"
#include "lpi_text.h"
#include "lpi_texture.h"
#include "lpi_screen.h"
#include "lpi_gui.h"
#include "lpi_pathfind.h"
#include "lodepng.h"
#include "lodewav.h"
#include "lpi_audio.h"
#include "lpi_math2d.h"

#include <vector>
#include <list>
#include <iostream>
#include <cmath>



using namespace lpi;

std::vector<Texture> textures;
std::vector<std::vector<double> > dead_sound;
std::vector<std::vector<double> > escape_sound;
Texture logo;

void loadTextures()
{
  loadTextures("textures.png", textures, TEXSIZEX, TEXSIZEY, AE_PinkKey);
  logo.create("logo.png");
}

void loadSounds()
{
  /*dead_sound.resize(20000);
  for(size_t i = 0; i < 20000; i++) dead_sound[i] = 0.3 * std::sin(i / 30.0);
  escape_sound.resize(10000);
  for(size_t i = 0; i < 10000; i++) escape_sound[i] = 0.3 * std::sin(i / (15.0 * (20000.0-i)/15000.0));*/
  
  std::vector<unsigned char> wav;
  LodeWAV::InfoWAV infowav;
  
  loadFile(wav, "escape.wav");
  LodeWAV::decode(escape_sound, infowav, wav);
  
  loadFile(wav, "dead.wav");
  LodeWAV::decode(dead_sound, infowav, wav);
}

bool Monster::calculatePath()
{
  PathFindPos start((int)pos.x, (int)pos.y)
            , end(world->endx, world->endy)
            , corner0(0, 0)
            , corner1(world->getsizex(), world->getsizey());
  testpath.clear();
  bool found = world->findPath(testpath, start, end, corner0, corner1);
  if(!found)
  {
    testpath.clear();
    testpath.push_back(PathFindPos(0, 0));
  }
  //else World_AStar::makeDirectionPath(testpath);
  return found;
}

const double SLOWTIME = 15.0;

void Monster::handle(double dt, double time)
{
  if(time - slowTime > SLOWTIME) slowed = 0; //after so many seconds, end the slowness
  
  if(path.empty()) return;
  
  if(!world->isInside(pos))
    pos = world->getRandomStartPos();
  
  double d = getSpeed() * dt; //distance to go
  double dd = d; //distance left
  
  if(path_index > path.size() - 1)
  {
    path_index = path.size() - 1; //normally this shouldn't actually occur, but ...
    return; //too far...
  }
  
  PathFindPos p(int(pos.x), int(pos.y));
  if(/*path_index == path.size() - 1*/p == path.back())
  {
    if(!deleted) deleted = escaped = true; //end is reached
    return;
  }
  
  if(deleted) return;
  
  for(;;)
  {
    Vector2 next(path[path_index + 1].x + 0.5, path[path_index + 1].y + 0.5); //the checkpoint

    double dist = distance(next, pos);
    Vector2 dir = (next - pos) / dist; //normalized
    
    if(dist > dd) //too far to reach with the distance we can walk this frame
    {
      pos += (dir * dd);
      break; //done
    }
    else
    {
      pos = next;
      path_index++;
      dd -= dist;
    }
  }
}

////////////////////////////////////////////////////////////////////////////////

void Tower::handle(double /*dt*/, double time)
{
  double shoottime = 1.0 / type->getRate(level);
  
  if(upgrading)
  {
    upgradePercentage = ((time - upgradeStartTime) / (upgradeEndTime - upgradeStartTime)) * 100.0;
    if(time >= upgradeEndTime)
    {
      upgrading = false;
      level++;
      
      shoottime = 1.0 / type->getRate(level); //recalculate since you upgraded
      lastTime = time - shoottime;
    }
    
    return;
  }
  
  if(!firstshot)
  {
    firstshot = true;
    lastTime = time - shoottime - 0.01;
  }
  
  if(time - lastTime > shoottime) //then try to shoot
  {
    lastTime += shoottime; //this makes the tower slower, but then it doesn't try to find a monster *every* frame
    
    bool slow = (type->effect == TowerType::SLOW);
    
    Monster* m = world->getMonsterNear(pos, type->getRange(level), slow);
    
    if(m) //then SHOOT
    {
      world->addBullet(pos, m->pos, this);
      if(type->method == TowerType::DIRECT)
        m->hit(type, level, time);
    }
  }
}

////////////////////////////////////////////////////////////////////////////////

bool World::testTowerPlacement(int x, int y)
{
  int old = cmap[x][y];
  cmap[x][y] = 1;
  bool test = calculateAllPaths();
  cmap[x][y] = old;
  return test;
}

bool World::calculateAllPaths()
{
  PathFindPos corner0(0, 0), corner1(getsizex(), getsizey());

  for(int fy = startfieldy0; fy < startfieldy1; fy++)
  for(int fx = startfieldx0; fx < startfieldx1; fx++)
  {
    if(!findPath(testpath, PathFindPos(fx, fy), PathFindPos(endx, endy), corner0, corner1))
    {
      return false;
    }
  }
  
  PathFindPos start;
  for(std::list<Monster*>::iterator it = monsters.begin(); it != monsters.end(); ++it)
  {
    start.x = int((*it)->pos.x);
    start.y = int((*it)->pos.y);
    if(!(*it)->calculatePath())
    {
      return false;
    }
  }
  return true;
}

void Bullet::handle(double dt, double time)
{
  current += dt / type->bullet_effect_speed;
  if(current > 1.0)
  {
    if(type->method == TowerType::SPLASHMISSILE)
      world->splashDamage(finish, type, level, time);
    deleted = true;
  }
}

////////////////////////////////////////////////////////////////////////////////

void WaveCreator::handle(double /*dt*/, double time, World* world, double health_modifier, double time_between_waves)
{
  if(timeTillNextWave(time, time_between_waves) <= 0.0)
  {
    createSpawn(world, health_modifier);
    lastMainSpawnTime = time;
    last_health = to_spawn.empty() ? 0 : to_spawn[0]->starthealth;
    level++;
  }
  
  if(to_spawn.empty()) return;
  
  if(time - lastSingleSpawnTime > TIMEBETWEENMONSTERS)
  {
    Monster* m = to_spawn.back();
    to_spawn.pop_back();
    m->pos = world->getRandomStartPos();
    m->calculatePath();
    m->useNewPath();
    world->monsters.push_back(m);
    lastSingleSpawnTime = time;
    //last_health = m->starthealth;
  }
}
  
////////////////////////////////////////////////////////////////////////////////

void printTowerStats(int x, int y, TowerType* t, int level, int printedlevel, const std::string& message, bool fullcost)
{
  std::stringstream ss;
  
  ss.clear(); ss.str("");
  ss << message;
  print(ss.str(), x, y);
  
  ss.clear(); ss.str("");
  ss << t->name;
  if(level > 0) ss << " " << printedlevel;
  print(ss.str(), x, y + 8, TS_Green);
  
  int cost;
  if(fullcost) cost = t->getFullCost(level);
  else cost = t->getCost(level);
  
  ss.clear(); ss.str("");
  ss << "cost: " << cost;
  print(ss.str(), x, y + 16);
  
  ss.clear(); ss.str("");
  ss << "damage: " << t->getDamage(level);
  print(ss.str(), x, y + 24);
  
  ss.clear(); ss.str("");
  ss << "range: " << t->getRange(level);
  print(ss.str(), x, y + 32);
  
  ss.clear(); ss.str("");
  ss << "rate: " << t->getRate(level);
  print(ss.str(), x, y + 40);
  
  ss.clear(); ss.str("");
  ss << "splash: " << t->getSplash(level);
  print(ss.str(), x, y + 48);
}

void printUserMonsterStats(int x, int y, MonsterType* t, Game& game, const std::string& message)
{
  std::stringstream ss;
  
  ss.clear(); ss.str("");
  ss << message;
  print(ss.str(), x, y);
  
  WaveCreator& waves = game.world[0].waves; //assuming it's the same for both players: pick one
  
  ss.clear(); ss.str("");
  ss << t->name;
  if(waves.level > 0) ss << " " << waves.level;
  print(ss.str(), x, y + 8, TS_Red);
  
  ss.clear(); ss.str("");
  ss << "cost: " << int(t->getUserWaveCost(waves.level, waves.lin_gold, waves.sq_gold, waves.exp_gold));
  print(ss.str(), x, y + 16);
  
  ss.clear(); ss.str("");
  ss << "health: " << t->getHealth(waves.level, waves.lin_health, waves.sq_health, waves.exp_health, waves.tweak_health, game.getHealthModifier());
  print(ss.str(), x, y + 24);
  
  ss.clear(); ss.str("");
  ss << "speed: " << t->speed;
  print(ss.str(), x, y + 32);
  
  ss.clear(); ss.str("");
  ss << "wavesize: " << t->wavesize;
  print(ss.str(), x, y + 40);
}

////////////////////////////////////////////////////////////////////////////////

void printFatalError(const std::string& text, bool newlines = false)
{
  cls();
  if(!newlines) printCentered(text, screenWidth() / 2, screenHeight() / 2);
  else print(text, 0, 0, TS_W, true);
  std::cout << text << std::endl;
  redraw();
  sleep();
}

void printPlayerStats(int x, int y, World& world, Game& game, double time)
{
  std::stringstream ss;
  
  ss.clear(); ss.str("");
  ss << "score: " << int(world.player.score);
  print(ss.str(), x, y + 0);
  
  ss.clear(); ss.str("");
  ss << "gold: " << int(world.player.gold);
  print(ss.str(), x, y + 8);
  
  ss.clear(); ss.str("");
  ss << "lives: " << world.player.lives;
  print(ss.str(), x, y + 16);
  
  ss.clear(); ss.str("");
  ss << "level: " << world.waves.level;
  print(ss.str(), x, y + 32);
  
  ss.clear(); ss.str("");
  ss << "juggle penalty: " << int(game.getJugglingPenalty());
  print(ss.str(), x, y + 40);
  
  ss.clear(); ss.str("");
  ss << "last monster's hp: " << int(world.waves.last_health);
  print(ss.str(), x, y + 48);
  
  if(game.active() && !world.waves.inPipeline())
  {
    ss.clear(); ss.str("");
    ss << "next wave: " << int(world.waves.timeTillNextWave(time, game.time_between_waves));
    print(ss.str(), x, y + 56);
  }
  
  Tower* worldtower = world.getTowerAt(world.controlx, world.controly);
  if(worldtower)
  {
    double sellprice = game.getSellingPrice(worldtower, world.waves.level, game.state == Game::PLAYING1);
    ss.clear(); ss.str("");
    ss << "sell price: " << int(sellprice);
    print(ss.str(), x, y + 64);
  }
}

void printTowerInfo(int x, int y, World& world, Game& game, int WORLDX, int WORLDY)
{
  Tower* worldtower = world.getTowerAt(world.controlx, world.controly);
  if(worldtower)
  {
    printTowerStats(x, y, worldtower->type, worldtower->level, worldtower->getPrintedLevel(), "current tower", true);
    if(!worldtower->isFullyUpgraded())
      printTowerStats(x + 120, y, worldtower->type, worldtower->level + 1, worldtower->getPrintedLevel() + 1, "upgrade", false);
    double radius = TEXSIZEX * worldtower->getRange();
    drawGradientDisk(WORLDX + worldtower->pos.x * TEXSIZEX, WORLDY + worldtower->pos.y * TEXSIZEY, radius, ColorRGB(255,255,255,128), ColorRGB(255,255,255,64));
  }
  else
  {
    TowerType* selectortower = world.player.selector.getOptionTower();
    if(selectortower)
    {
      printTowerStats(x, y, selectortower, 0, 1, "new tower", false);
      double radius = TEXSIZEX * selectortower->getRange(0);
      drawGradientDisk(WORLDX + (world.controlx + 0.5) * TEXSIZEX, WORLDY + (world.controly + 0.5) * TEXSIZEY, radius, ColorRGB(255,224,224,128), ColorRGB(255,224,224,64));
    }
  }
  
  //monster waves to send
  MonsterType* selectormonster = world.player.selector.getOptionMonster();
  if(selectormonster)
  {
    printUserMonsterStats(x + 240, y, selectormonster, game, "attack");
  }
  
}

////////////////////////////////////////////////////////////////////////////////

int main(int, char*[]) //the arguments have to be given here, or DevC++ can't link to SDL for some reason
{
  //test_pathfind();
  
  screen(800, 600, 0, "Torenoorlog");
  loadTextures();
  
  if(!getFilesize("textures.png") || !getFilesize("logo.png") || !getFilesize("tw.txt") || !getFilesize("controls.txt"))
  {
    printFatalError("One or more data files not found.\n\
This game needs following files in its directory or the pwd:\n\
*) textures.png: the game textures, at least 240x240 pixels\n\
*) logo.png: the Torenoorlog logo\n\
*) tw.txt: xml code describing the <monstertype>s and <towertype>s\n\
*) controls.txt: xml code describing the user keys\n\
*) dead.wav: sound when monster dies (mono, 44100Hz, 16-bit)\n\
*) escape.wav: sound when monster escapes (mono, 44100Hz, 16-bit)", true);
    return 0;
  }
  
  Game game;
  size_t pos, parseresult;
  
  std::string xml;
  loadFile(xml, "tw.txt");
  pos = 0;
  parseresult = xml::parse(game, xml, pos);
  if(xml::isError(parseresult))
  {
    std::stringstream ss;
    ss << "XML parsing error " << parseresult << " on or before line " << xml::getLineNumber(xml, pos) << " in tw.txt";
    printFatalError(ss.str());
    return 0;
  }
  
  loadFile(xml, "controls.txt");
  pos = 0;
  parseresult = game.keys.parse(xml, pos);
  if(xml::isError(parseresult))
  {
    std::stringstream ss;
    ss << "XML parsing error " << parseresult << " on or before line " << xml::getLineNumber(xml, pos) << " in controls.txt";
    printFatalError(ss.str());
    return 0;
  }
  
  gui::Container guiContainer;
  
  gui::Window pregameWindow;
  pregameWindow.make(640, 440, 200, 240);
  pregameWindow.addTop();
  pregameWindow.addTitle("New Game");
  guiContainer.pushTop(&pregameWindow);
  pregameWindow.moveCenterTo(400, 320);
  pregameWindow.setColor(ColorRGB(192, 192, 255, 224));
  
  gui::Window gameWindow;
  gameWindow.make(630, 4, 146, 88);
  gameWindow.addTop();
  gameWindow.addTitle("Game Controls");
  gameWindow.setColor(ColorRGB(192, 192, 255, 224));
  guiContainer.pushTop(&gameWindow);
  
  gui::Window gameoverWindow;
  gameoverWindow.make(640, 440, 300, 128);
  gameoverWindow.addTop();
  gameoverWindow.addTitle("Game Over");
  guiContainer.pushTop(&gameoverWindow);
  gameoverWindow.setColor(ColorRGB(255, 192, 192, 224));
  gameoverWindow.moveCenterTo(400, 320);
  
  gui::Window aboutWindow;
  aboutWindow.make(0, 0, 400, 200);
  aboutWindow.moveCenterTo(400, 300);
  gui::FormattedText abouttext;
  abouttext.make(0, 0, "Torenoorlog#n\
Tower defense for two#nversion: 20071208#n#n\
Credits:#n#n\
-programming: Lode Vandevenne#n\
-default art: Lode Vandevenne with KolourPaint#n\
-default sounds: http://freesound.iua.upf.edu/#n\
-mod: " + game.mod_author);
  aboutWindow.addTop();
  aboutWindow.addTitle("About");
  aboutWindow.setColor(ColorRGB(255, 192, 192, 224));
  aboutWindow.pushTopAt(&abouttext, 8, 8);
  aboutWindow.totallyDisable();
  aboutWindow.addCloseButton();
  aboutWindow.close();
  guiContainer.pushTop(&aboutWindow);
  
  //single or two player
  gui::BulletList numplayers;
  numplayers.make(0, 0, 2, 0, 16);
  numplayers.addText("single player practice", 0);
  numplayers.addText("two player war", 1);
  numplayers.set(1);
  pregameWindow.pushTopAt(&numplayers, 8, 4);
  
  //difficulty
  gui::BulletList difficulty;
  difficulty.make(0, 0, 5, 0, 16);
  difficulty.addText("sandbox (practice)", 0);
  difficulty.addText("easy", 1);
  difficulty.addText("normal", 2);
  difficulty.addText("hard", 3);
  difficulty.addText("nightmare", 4);
  difficulty.set(2);
  pregameWindow.pushTopAt(&difficulty, 8, 46);
  
  //an exit button on each window...
  gui::Button exitbutton0;
  exitbutton0.makeTextPanel(0, 0, "exit");
  pregameWindow.pushTopAt(&exitbutton0, 8, 192);
  gui::Button exitbutton1;
  exitbutton1.makeTextPanel(0, 0, "exit");
  gameWindow.pushTopAt(&exitbutton1, 76, 40);
  gui::Button exitbutton2;
  exitbutton2.makeTextPanel(0, 0, "exit");
  gameoverWindow.pushTopAt(&exitbutton2, 8, 82);
 
  gui::Button startbutton;
  startbutton.makeTextPanel(0, 0, "start");
  pregameWindow.pushTopAt(&startbutton, 8, 128);
  
  gui::Button aboutbutton;
  aboutbutton.makeTextPanel(8, 40, "about");
  pregameWindow.pushTopAt(&aboutbutton, 8, 160);
  
  gui::Button nextbutton;
  nextbutton.makeTextPanel(0, 0, "start"); //becomes "next" in PLAYING1 phase
  gameWindow.pushTopAt(&nextbutton, 4, 8);
  
  gui::Button restartbutton;
  restartbutton.makeTextPanel(0, 0, "new game");
  gameWindow.pushTopAt(&restartbutton, 76, 8);
  
  gui::Button restartbutton_gameover;
  restartbutton_gameover.makeTextPanel(0, 0, "new game");
  gameoverWindow.pushTopAt(&restartbutton_gameover, 8, 50);
  
  gui::Button pausebutton;
  pausebutton.makeTextPanel(0, 0, "pause");
  gameWindow.pushTopAt(&pausebutton, 4, 40);
  
  gui::FormattedText gameovertext;
  gameovertext.make();
  gameoverWindow.pushTopAt(&gameovertext, 8, 8);
  
  gui::Checkbox audiobox;
  audiobox.make(0, 0, false);
  audiobox.addText("sounds");
  guiContainer.pushTopAt(&audiobox, 720, 580);
  
  
  GameTime gametime;
  
  double time = 0.0;
  double deltatime = 0.0;
  
  const int WORLDX[2] = {0, 400};
  const int WORLDY[2] = {100, 100};
  
  game.world[0].setStartAndEnd(15, 6, 16, 10, 0, 0);
  game.world[1].setStartAndEnd(0, 6, 1, 10, 15, 0);
  game.reset();
  
  guiContainer.pushTop(&game.world[0]);
  guiContainer.pushTop(&game.world[1]);
  game.world[0].setNotDrawnByContainer(true);
  game.world[1].setNotDrawnByContainer(true);

  for(size_t p = 0; p < MAXNUMPLAYERS; p++)
  {
    game.world[p].testTowerPlacement();
    game.world[p].useNewPaths();
    game.world[p].waves.createWaveOrder(game.monstertypes);
    
    game.world[p].player.selector.moveTo(WORLDX[p], WORLDY[p] - 24);
    game.world[p].moveTo(WORLDX[p], WORLDY[p]);
  }
  
  loadSounds();
  
  audioOpen(44100, 2048);
  
  //////////////////////////////////////////////////////////////////////////////
  
  while(frame(false, true)) //the gameloop
  {
    ///update time
    gametime.update();
    if(game.active())
    {
      time += gametime.diff_s();
      deltatime = gametime.diff_s();
    }
    else deltatime = 0.0;
    
    ///show the window depending on the game mode and handle gui
    if(game.state == Game::PREGAME)
    {
      pregameWindow.totallyEnable();
      gameWindow.totallyDisable();
      gameoverWindow.totallyDisable();
    }
    else if(game.state == Game::PLAYING0 || game.state == Game::PLAYING1)
    {
      pregameWindow.totallyDisable();
      gameWindow.totallyEnable();
      gameoverWindow.totallyDisable();
    }
    else if(game.state == Game::GAMEOVER)
    {
      pregameWindow.totallyDisable();
      gameWindow.totallyDisable();
      gameoverWindow.totallyEnable();
    }
    
    if(game.state == Game::PREGAME)
    {
      game.world[0].totallyDisable();
      game.world[1].totallyDisable();
    }
    else if(game.getNumPlayers() == 1)
    {
      game.world[0].totallyEnable();
      game.world[1].totallyDisable();
    }
    else
    {
      game.world[0].totallyEnable();
      game.world[1].totallyEnable();
    }
    
    ///gamestate depending gui and user input things
    if(game.state == Game::PREGAME)
    {
      if(startbutton.clicked() || keyPressed(game.keys.next[0])) game.state = Game::PLAYING0;
      
      if(difficulty.check() == 0) game.difficulty = Game::SANDBOX;
      else if(difficulty.check() == 1) game.difficulty = Game::EASY;
      else if(difficulty.check() == 2) game.difficulty = Game::NORMAL;
      else if(difficulty.check() == 3) game.difficulty = Game::HARD;
      else if(difficulty.check() == 4) game.difficulty = Game::NIGHTMARE;
      
      if(numplayers.check() == 0) game.mode = Game::SINGLEPLAYER;
      else if(numplayers.check() == 1) game.mode = Game::TWOPLAYER;
      
      game.reset();
    }
    else if(game.state == Game::PLAYING0)
    {
      if(nextbutton.clicked() || game.bothPlayersWantToNextLevel()) game.state = Game::PLAYING1;
      if(restartbutton.clicked()) game.state = Game::PREGAME;
    }
    else if(game.state == Game::PLAYING1)
    {
      if(nextbutton.clicked() || game.bothPlayersWantToNextLevel())
      {
        for(size_t p = 0; p < game.getNumPlayers(); p++)
        {
          if(!game.world[p].waves.inPipeline())
            game.world[p].waves.spawnNow();
        }
      }
      if(restartbutton.clicked()) game.state = Game::PREGAME;
    }
    else if(game.state == Game::GAMEOVER)
    {
      if(restartbutton_gameover.clicked()) game.state = Game::PREGAME;
    }
    
    if(aboutbutton.clicked()) { guiContainer.pushTop(&aboutWindow); aboutWindow.unClose(); }
    if(aboutWindow.closeButton.pressed()) aboutWindow.close();
    
    if(exitbutton0.clicked()) return 0;
    if(exitbutton1.clicked()) return 0;
    if(exitbutton2.clicked()) return 0;
    
    if(game.state == Game::PLAYING0) nextbutton.text = "start"; else nextbutton.text = "next";
    
    if(game.paused) pausebutton.text = "resume"; else pausebutton.text = "pause";
    if(pausebutton.clicked()) game.paused = !game.paused;
    
    if(audiobox.isChecked()) audioSetMode(1);
    else audioSetMode(0);
    
    ///run the game
    
    if(game.state == Game::PLAYING1)
    {
      for(size_t p = 0; p < game.getNumPlayers(); p++)
      {
        game.world[p].waves.handle(deltatime, time, &game.world[p], game.getHealthModifier(), game.time_between_waves);
      }
    }
    
    if(game.state != Game::PREGAME)
    {
    
      //player 0
      {
        game.world[0].draw();
        game.world[0].player.selector.monsters_ready = game.getNumPlayers() > 1 && game.world[0].player.canSpawnUserWave(time, game.time_between_waves);
        game.world[0].player.selector.draw();
        game.world[0].player.selector.handle();
        game.world[0].player.message.draw(192, 292, gametime.get_s());
      }
      
      //player 1
      if(game.getNumPlayers() > 1)
      {
        game.world[1].draw();
        game.world[1].player.selector.monsters_ready = game.getNumPlayers() > 1 && game.world[1].player.canSpawnUserWave(time, game.time_between_waves);
        game.world[1].player.selector.draw();
        game.world[1].player.selector.handle();
        game.world[1].player.message.draw(592, 292, gametime.get_s());
      }
      
      //more player 0
      {
        printPlayerStats(0, 0, game.world[0], game, time);
        printTowerInfo(0, 500, game.world[0], game, WORLDX[0], WORLDY[0]);
      }
      //more player 1
      if(game.getNumPlayers() > 1)
      {
        printPlayerStats(400, 0, game.world[1], game, time);
        printTowerInfo(400, 500, game.world[1], game, WORLDX[1], WORLDY[1]);
      }
      
      //general stuff
      print(difficulty.getCurrentText() + " mode", 0, 580);
      print(numplayers.getCurrentText(), 0, 588);
    }
    else
    {
      logo.drawCentered(400, 150);
    }
    
    if(game.state == Game::PLAYING0 || game.state == Game::PLAYING1)
    {
      game.world[0].handle(deltatime, time);
      if(game.getNumPlayers() > 1) game.world[1].handle(deltatime, time);
    }
    
    
    guiContainer.draw();
    guiContainer.handle();
   
    redraw();
    cls(ColorRGB(64, 64, 64));
    
    if(!game.paused && game.state != Game::PREGAME)
    for(size_t p = 0; p < game.getNumPlayers(); p++)
    {
      if(keyPressedTime(game.keys.left[p], gametime.get_s(), 0.5, 0.05)) game.world[p].controlx--;
      if(keyPressedTime(game.keys.right[p], gametime.get_s(), 0.5, 0.05)) game.world[p].controlx++;
      if(keyPressedTime(game.keys.up[p], gametime.get_s(), 0.5, 0.05)) game.world[p].controly--;
      if(keyPressedTime(game.keys.down[p], gametime.get_s(), 0.5, 0.05)) game.world[p].controly++;
      game.world[p].wrapController();
      if(keyPressedTime(game.keys.selectleft[p], gametime.get_s(), 0.5, 0.05)) game.world[p].player.selector.sel--;
      if(keyPressedTime(game.keys.selectright[p], gametime.get_s(), 0.5, 0.05)) game.world[p].player.selector.sel++;
      game.world[p].player.selector.wrapController();
      if(keyPressed(game.keys.next[p]))
      {
        game.world[p].wants_to_next_level = game.world[p].waves.level;
      }
      if(keyPressed(game.keys.exec[p]))
      {
        switch(game.world[p].player.selector.getOptionType())
        {
          case Selector::TOWER:
            if(game.state != Game::GAMEOVER && game.world[p].cmap[game.world[p].controlx][game.world[p].controly] != 1)
            {
              bool doesntblock = game.world[p].testTowerPlacement();
              bool enough_gold = false;
              bool placable = game.world[p].towerplacable();
              
              TowerType* type = game.world[p].player.selector.getOptionTower();
              
              enough_gold = (game.world[p].player.gold >= type->getCost());
              
              if(doesntblock && enough_gold && placable)
              {
                game.world[p].useNewPaths();
                game.world[p].cmap[game.world[p].controlx][game.world[p].controly] = 1;
                
                Tower* t = new Tower(&game.world[p], type);
                t->pos.x = game.world[p].controlx + 0.5;
                t->pos.y = game.world[p].controly + 0.5;
                game.world[p].towers.push_back(t);
                t->type = type;
                
                game.world[p].player.gold -= type->getCost();
              }
  
              if(!doesntblock) game.world[p].player.message.set("blocking!");
              else if(!enough_gold) game.world[p].player.message.set("not enough gold");
              else if(!placable) game.world[p].player.message.set("can't build here");
            }
            break;
          case Selector::MONSTER:
            {
              if(game.state == Game::PLAYING1 && game.world[p].player.canSpawnUserWave(time, game.time_between_waves) && game.getNumPlayers() > 1)
              {
                WaveCreator& waves = game.world[p].waves;
                MonsterType* type = game.world[p].player.selector.getOptionMonster();
                double cost = type->getUserWaveCost(waves.level, waves.lin_gold, waves.sq_gold, waves.exp_gold);
                bool enough_gold = (game.world[p].player.gold >= cost);
                
                if(enough_gold)
                {
                  int otherplayer = p == 0 ? 1 : 0;
                  game.world[otherplayer].waves.createSpawn(&game.world[otherplayer], type, game.getHealthModifier());
                  game.world[p].player.lastUserWaveTime = time;
                  game.world[p].player.gold -= cost;
                }
                else game.world[p].player.message.set("not enough gold");
              }
            }
            break;
          default: break;
        }
      }
      if(keyPressed(game.keys.sell[p]))
      {
        Tower* t = game.world[p].getTowerAt(game.world[p].controlx, game.world[p].controly);
        if(t)
        {
          double gold = game.getSellingPrice(t, game.world[p].waves.level, game.state == Game::PLAYING1);
          if(game.world[p].player.gold + gold >= 0)
          {
            game.world[p].player.gold += gold;
            game.world[p].deleteTowerAt(game.world[p].controlx, game.world[p].controly);
            game.world[p].cmap[game.world[p].controlx][game.world[p].controly] = 0;
            game.world[p].calculateAllPaths();
            game.world[p].useNewPaths();
          }
          else
          {
            game.world[p].player.message.set("not enough gold to pay juggling penalty");
          }
        }
      }
      if(keyPressed(game.keys.upgrade[p]))
      {
        Tower* t = game.world[p].getTowerAt(game.world[p].controlx, game.world[p].controly);
        if(t && !t->upgrading && !t->isFullyUpgraded())
        {
          int gold = t->getUpgradePrice();
          if(game.world[p].player.gold - gold >= 0)
          {
            game.world[p].player.gold -= gold;
            t->startUpgrade(time, game.state == Game::PLAYING1);
          }
          else
          {
            game.world[p].player.message.set("not enough gold");
          }
        }
      }
      
      if(game.world[p].player.lives <= 0)
      {
        game.state = Game::GAMEOVER;
        game.world[p].player.message.set("Game Over");
        guiContainer.pushTop(&gameoverWindow);
        
        std::stringstream ss;
        ss << difficulty.getCurrentText() << " mode" << "#n";
        if(game.mode == Game::SINGLEPLAYER)
        {
          ss << "score: " << int(game.world[0].player.score) << "#n";
        }
        else if(game.mode == Game::TWOPLAYER)
        {
          ss << "player " << (p == 0 ? 2 : 1) << " won!#n";
          ss << "player 1 score: " << int(game.world[0].player.score) << "#n";
          ss << "player 2 score: " << int(game.world[1].player.score) << "#n";
        }
        gameovertext.setText(ss.str());
      }
    } //end of "per player" loop
    
    if(game.paused)
    {
      game.world[0].player.message.set("Paused");
      game.world[1].player.message.set("Paused");
    }
    
    
    if(keyDown('i') && keyDown('l') && keyPressed('m') && nextbutton.mouseDown()) //cheat code
    {
      game.world[0].player.message.set("CHEAT CODE ACTIVATED");
      game.world[0].player.lives += 100;
      game.world[0].player.gold += 10000;
      
      game.world[1].player.message.set("CHEAT CODE ACTIVATED");
      game.world[1].player.lives += 100;
      game.world[1].player.gold += 10000;
    }
  }
  
  
  return 0;
}



