/*
LLL Programming Language Interpreter

Copyright (c) 2012 Lode Vandevenne
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
  * Neither the name of Lode Vandevenne nor the names of his contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#pragma once

#include "main.h"

#include <cassert>

extern bool initphase; //initialization phase is busy

class Part
{
  public:
    std::vector<Connector> c; //parts it's connected to.
    bool initialized;
    
    Part() : initialized(false) {}
    virtual ~Part(){}

  /*
  Calculate your state.
  this is called on all parts, and the intention is that you always calculate the next state using the previous (aka current) state of all neighbor parts.
  To achieve that, first calculate is called on all parts, and they should store the result internally.
  Next, update is called on all parts, where they put the calculated next state, into their current state.
  The calculation should happen by calling "powered" on all connections, and using their results with your logic to set the state.
  For example, if this is a xor wire, and a connection is a diode, then the diode's "powered" function is responsible for telling this xor wire that it received
  power from that connection only if this is the correct side of the diode AND the diode is powered up by its input. But the xor wire is responsible for doing
  the "xor" calculation, and using that in turn for "powered" calls from the other parts connected to this xor wire.
  The component, in its calculation, should not call "powered" on connections that don't make sense, e.g. the diode should call powered only on the part connected
  to its input, not to its output.
  */
  virtual void calculate() = 0;
  
  /*
  With this, put the calculated next state, as the current state.
  */
  virtual void update() = 0;
  
  /*
  Initialize before program start. Only used by invertors for their rule: every
  invertor which receives true at its input before program start, is turned off,
  to fix warmup-pulses from AND gates.
  */
  virtual void initialize()
  {
    if(!initialized)
    {
      initialized = true;
      calculate();
      update();
    }
  }

  /*
  Does this component give power to the given connected component?
  pos = in connector coordinates
  */
  virtual bool powered(const Part* p, const Pos& pos)
  {
    if(!initialized) initialize();
    return poweredImpl(p, pos);
  }

  /*
  Does this component give power to the given connected component?
  pos = in connector coordinates
  */
  virtual bool poweredImpl(const Part* p, const Pos& pos) = 0;
};

class StatedPart : public Part
{
  public:
    bool state;
    bool nextstate;
    
    StatedPart() :  state(false), nextstate(false) {}
    
    virtual void update()
    {
      state = nextstate;
    }
    
    virtual bool poweredImpl(const Part* p, const Pos& pos)
    {
      (void)p; (void)pos;
      return state;
    }
};

class InstantPart : public Part
{
  public:
    bool calculated;
    
    InstantPart() : calculated(false) {}
    
    virtual void calculate()
    {
      //does nothing. For being "instant", so the calculation is done in "powered" instead.
    }
    
    virtual void calculate2() = 0;
    
    virtual void update()
    {
      //reset for next frame
      calculated = false;
    }
    
    virtual bool poweredImpl(const Part* p, const Pos& pos)
    {
      (void)p; (void)pos;
      if(!calculated) calculate2();
      calculated = true;
      return poweredImpl2(p, pos);
    }
    
    virtual bool poweredImpl2(const Part* p, const Pos& pos) = 0;
};

class Wire : public InstantPart
{
  public:
    bool state;
    
    Wire() : state(false) {}
    
    void calculate2()
    {
      state = false;
      for(size_t i = 0; i < c.size(); i++)
      {
        if(c[i].other->powered(this, c[i].pos))
        {
          state = true;
          break;
        }
      }
    }
    
    virtual bool poweredImpl2(const Part* p, const Pos& pos)
    {
      (void)p; (void)pos;
      return state;
    }
};

class XorWire : public InstantPart
{
  public:
    bool state;
    
    XorWire() : state(false) {}
    
    virtual void calculate2()
    {
      state = false;
      for(size_t i = 0; i < c.size(); i++)
      {
        if(c[i].other->powered(this, c[i].pos)) state = !state;
      }
    }
    
    virtual bool poweredImpl2(const Part* p, const Pos& pos)
    {
      (void)p; (void)pos;
      return state;
    }
};

class CounterWire : public InstantPart
{
  public:
    bool previn;
    bool state;
    
    CounterWire() : previn(false), state(false) {}
    
    virtual void calculate2()
    {
      bool in = false;
      for(size_t i = 0; i < c.size(); i++)
      {
        if(c[i].other->powered(this, c[i].pos)) in = true;
      }
      if(in && !previn) state = !state;
      previn = in;
      if(initphase)
      {
        state = false; //initially always off
        previn = true;
      }
    }
    
    virtual bool poweredImpl2(const Part* p, const Pos& pos)
    {
      (void)p; (void)pos;
      return state;
    }
};

class DelayWire : public InstantPart
{
  public:
    size_t last_on;
    size_t tick;
    size_t n;
    
    DelayWire() : last_on(0), tick(0), n(0) {}
    
    virtual void update()
    {
      InstantPart::update();
      tick++;
    }
    
    virtual void calculate2()
    {
      bool in = false;
      for(size_t i = 0; i < c.size(); i++)
      {
        if(c[i].other->powered(this, c[i].pos)) in = true;
      }
      if(in) last_on = tick;
    }
    
    virtual bool poweredImpl2(const Part* p, const Pos& pos)
    {
      (void)p;
      (void)pos;
      return tick - last_on <= n;
    }
};

class IO : public StatedPart
{
  public:
    int num; //the input or output channel number
    bool numbered;
    bool highnumbered; //true: it's a high numbered signal, false: it's a bit
    std::vector<Pos> positions; //all positions in which this input or output is located
    
    
    IO() : num(-1), numbered(false), highnumbered(false) {}
};

class Input : public IO
{
  public:
    bool mouseOverride; //once you click on it in interactive SDL mode, numbered inputs become toggled inputs
    int x0, y0, x1, y1; //bounding box for faster mouse click detection (endings excluded)
    bool button; //behaves like a button instead of a toggle switch in interactive mode
    
    Input() : mouseOverride(false), button(false) {}
    
    void calcBB()
    {
      bool inited = false;
      for(size_t i = 0; i < positions.size(); i++)
      {
        int x = positions[i].x;
        int y = positions[i].y;
        if(!inited)
        {
          x0 = x;
          x1 = x + 1;
          y0 = y;
          y1 = y + 1;
          inited = true;
        }
        if(x < x0) x0 = x;
        if(x >= x1) x1 = x + 1;
        if(y < y0) y0 = y;
        if(y >= y1) y1 = y + 1;
      }
    }
    
    bool mouseInside()
    {
      int mx = mouseX / pixelsize;
      int my = mouseY / pixelsize;
      if(mx >= x0 && mx < x1 && my >= y0 && my < y1)
      {
        for(size_t i = 0; i < positions.size(); i++)
        {
           if(positions[i].x == mx && positions[i].y == my) return true;
        }
      }
      return false;
    }
    
    virtual void calculate()
    {
#ifdef SDLGRAPHICS
      if(button)
      {
        if(LMB && mouseInside())
        {
          mouseOverride = true;
          nextstate = true;
        }
        else
        {
          nextstate = false;
        }
      }
      else
      {
        bool clicked = mouseJustDownCircuit && mouseInside();
        if(clicked) mouseOverride = true;
        if(clicked) nextstate = !state;
      }

      if(!mouseOverride)
      {
#endif
        if(numbered)
        {
          if(highnumbered)
          {
            if(num == 0)
            {
              nextstate = io.in_eof;
            }
            else if(num == 1)
            {
              nextstate = io.in_update;
            }
          }
          else
          {
            assert(num >= 0 && num <= 7);
            nextstate = (io.in >> num) & 1;
          }
        }
#ifdef SDLGRAPHICS
      }
#endif
    }
    
  //overridden because inputs always return false during initialization phase
  virtual bool powered(const Part* p, const Pos& pos)
  {
    if(!initialized) return false;
    return poweredImpl(p, pos);
  }

  virtual void initialize()
  {
  }
  
  virtual void update()
  {
    initialized = true;
    StatedPart::update();
  }
};

class Output : public IO
{
  public:
    virtual void calculate()
    {
      nextstate = false;
      for(size_t i = 0; i < c.size(); i++)
      {
        if(c[i].other->powered(this, c[i].pos)) nextstate = true;
      }
      if(numbered)
      {
        if(highnumbered)
        {
          if(num == 0)
          {
            io.out_eof |= nextstate;
          }
          else if(num == 1)
          {
            io.out_update |= nextstate;
          }
          else if(num == 2)
          {
            io.out_inputready |= nextstate;
          }
        }
        else
        {
          assert(num >= 0 && num <= 7);
          if(nextstate) io.out |= (1 << num);
        }
      }
    }
    
    virtual bool poweredImpl(const Part* p, const Pos& pos)
    {
      (void)p;
      (void)pos;
      return false; //output transits its state to NOTHING
    }
};

class DirectedStatedPart : public StatedPart
{
  public:
  
    Pos outputConnector; //in connector coordinates
    Pos pos;
    
    DirectedStatedPart(){}
    
    virtual bool poweredImpl(const Part* p, const Pos& pos)
    {
      (void)p;
      return state && (pos == outputConnector);
    }

    //overridden to initialize only if question comes from the correct direction
    //if not done, a XOR gate with invertors before and after is still not
    //initialized in a correct state!!
    virtual bool powered(const Part* p, const Pos& pos)
    {
      if(!initialized && pos == outputConnector) initialize();
      return poweredImpl(p, pos);
    }
};

class Diode : public DirectedStatedPart
{
  public:
    Diode() {}
    
    virtual void calculate()
    {
      nextstate = false;
      for(size_t i = 0; i < c.size(); i++)
      {
        if(c[i].pos == outputConnector) continue;
        if(c[i].other->powered(this, c[i].pos)) nextstate = true;
        break;
      }
    }
};

class Invertor : public DirectedStatedPart
{
  public:
    Invertor()
    {
      state = true;
    }
    
    virtual void calculate()
    {
      nextstate = true;
      for(size_t i = 0; i < c.size(); i++)
      {
        if(c[i].pos == outputConnector) continue;
        if(c[i].other->powered(this, c[i].pos)) nextstate = false;
        break;
      }
    }
};

