/*
LLL Programming Language Interpreter

Copyright (c) 2012 Lode Vandevenne
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
  * Neither the name of Lode Vandevenne nor the names of his contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "parse.h"
#include "main.h"

#include <cassert>
#include <iostream>
#include <fstream>
#include <cassert>
#include <cstdlib>

#include "parts.h"

/*
Turns an index of 0-7 into a shift of x and y coordinates. The order is:
N, E, S, W, NW, NE, SE, SW
Returns true if result is in bounds, false if result is out of bounds.
*/
bool innerIndexToPos(int& x, int& y, int i, size_t w, size_t h)
{
  if(i == 0 || i == 4 || i == 5) y--;
  if(i == 1 || i == 5 || i == 6) x++;
  if(i == 2 || i == 6 || i == 7) y++;
  if(i == 3 || i == 7 || i == 4) x--;
  return (x >= 0 && y >= 0 && x < (int)w && y < (int)h);
}

/*
Is cell fully done?
*/
bool cellDone(int x, int y,
              std::vector<bool>& done0, std::vector<bool>& done1,
              const std::vector<char>& in, size_t w, size_t h)
{
  (void)h;
  size_t index = y * w + x;
  char c = in[index];
  if(c == 'x' || c == '+') return done0[index] && done1[index];
  else return done0[index];
}

/*
Given where you're now (ox, oy) and where this cell is (cx, cy), is the part
of the cell you're connected to already done?
ox, oy must touch cx, cy at corner or side.
*/
bool cellDone(int cx, int cy, int ox, int oy,
              std::vector<bool>& done0, std::vector<bool>& done1,
              const std::vector<char>& in, size_t w, size_t h)
{
  (void)h;
  size_t index = cy * w + cx;
  char c = in[index];
  if(c == 'x')
  {
    if(ox - cx == oy - cy) return done0[index];
    else return done1[index];
  }
  else if(c == '+')
  {
    if(ox == cx) return done0[index];
    else return done1[index];
  }
  else return done0[index];
}

void markDone(int cx, int cy, int d,
              std::vector<bool>& done0, std::vector<bool>& done1,
              size_t w, size_t h)
{
  (void)h;
  size_t index = cy * w + cx;
  if(d) done1[index] = true;
  else done0[index] = true;
}

void markDone(int cx, int cy, int ox, int oy,
              std::vector<bool>& done0, std::vector<bool>& done1,
              const std::vector<char>& in, size_t w, size_t h)
{
  (void)h;
  size_t index = cy * w + cx;
  char c = in[index];
  int d = 0;
  if(c == 'x') d = (ox - cx != oy - cy);
  if(c == '+') d = (ox != cx);
  markDone(cx, cy, d, done0, done1, w, h);
}

/*
add cell to stack and mark the correct done bit
does nothing if it's already marked as done
*/
void addAndMark(std::vector<Pos>& stack,
                int cx, int cy, int ox, int oy,
                std::vector<bool>& done0, std::vector<bool>& done1,
                const std::vector<char>& in, size_t w, size_t h)
{
  if(cellDone(cx, cy, ox, oy, done0, done1, in, w, h)) return;
  markDone(cx, cy, ox, oy, done0, done1, in, w, h);
  (void)h;
  size_t index = cy * w + cx;
  char c = in[index];
  int d = 0;
  if(c == 'x') d = (ox - cx != oy - cy);
  if(c == '+') d = (ox != cx);
  stack.push_back(Pos(cx, cy, d));
}

void addAndMark(std::vector<Pos>& stack,
                int x, int y, int d,
                std::vector<bool>& done0, std::vector<bool>& done1,
                size_t w, size_t h)
{
  markDone(x, y, d, done0, done1, w, h);
  stack.push_back(Pos(x, y, d));
}

/*
"from to" coordinates to "connector coordinates"
the from and to coordinate must tuch at a side
*/
Pos fromToToCoCo(int xto, int yto, int xfrom, int yfrom)
{
  Pos result;
  if(yto == yfrom)
  {
    result.d = 1;
    result.y = yfrom;
    result.x = std::max(xto, xfrom);
  }
  else
  {
    result.d = 0;
    result.x = xfrom;
    result.y = std::max(yto, yfrom);
  }
  return result;
}

void addToConnectorMap(ConnectorMap& map,
                       int xto, int yto, int xfrom, int yfrom, Part* part)
{
  Pos pos = fromToToCoCo(xto, yto, xfrom, yfrom);
  TwoParts& p = map[pos];
  if(pos.d == 1)
  {
    p[xfrom < xto ? 0 : 1] = part;
  }
  else
  {
    p[yfrom < yto ? 0 : 1] = part;
  }
}

void resolveConnectors(ConnectorMap& map)
{
  for(ConnectorMap::iterator it = map.begin(); it != map.end(); ++it)
  {
    if(!it->second[0] || !it->second[1])
    {
      std::cout << "error at: " << it->first.x << " " << it->first.y << " " << it->first.d << std::endl;
      assert(it->second[0]);
      assert(it->second[1]);
    }
    it->second[0]->c.push_back(Connector(it->first, it->second[1]));
    it->second[1]->c.push_back(Connector(it->first, it->second[0]));
  }
}

/*
parse a wire part
*/
Part* parseWire(std::vector<Connector>& stack, ConnectorMap& map,
                const Pos& start,
                std::vector<bool>& done0, std::vector<bool>& done1,
                const std::vector<char>& in, size_t w, size_t h)
{
  (void)stack;
  std::vector<Pos> stack2;
  addAndMark(stack2, start.x, start.y, start.d, done0, done1, w, h);
  Wire* result = new Wire();
  while(!stack2.empty())
  {
    Pos pos = stack2.back();
    stack2.pop_back();
    size_t index = pos.y * w + pos.x;
    char c = in[index];
    
    if(c == 'b')
    {
      for(size_t y3 = 0; y3 < h; y3++)
      for(size_t x3 = 0; x3 < w; x3++)
      {
        if((int)x3 == pos.x && (int)y3 == pos.y) continue;
        size_t index3 = y3 * w + x3;
        char c3 = in[index3];
        if(c3 == 'b')
        {
          //bool notdone3 = !cellDone(x3, y3, done0, done1, in, w, h);
          //assert((x3 == pos.x && y3 == pos.y) || notdone3);
          addAndMark(stack2, x3, y3, 0, 0, done0, done1, in, w, h);
        }
      }
    }

    for(int i = 0; i < 8; i++)
    {
      if(c == '+' && (i >= 4 || i % 2 != pos.d)) continue;
      if(c == 'x' && (i < 4 || i % 2 != pos.d)) continue;
      if(c == 'b' && i >= 4) continue;
      
      int x2 = pos.x, y2 = pos.y;
      if(!innerIndexToPos(x2, y2, i, w, h)) continue;
      
      size_t index2 = y2 * w + x2;
      char c2 = in[index2];
      if(c == '*' && ((i >= 4 && c2 != 'x') || (i < 4 && c2 == 'x'))) continue;
      if(c == 'x' && !(c2 == '*' || c2 == 'x')) continue;
      
      if(c2 == '*' || c2 == '+' || c2 == 'x')
      {
        if(c == '+' && c2 == 'x') continue;
        if(c == 'x' && c2 == '+') continue;
        addAndMark(stack2, x2, y2, pos.x, pos.y, done0, done1, in, w, h);
      }
      else if(c2 == '^' || c2 == '>' || c2 == 'v' || c2 == '<'
           || c2 == 'm' || c2 == ']' || c2 == 'w' || c2 == '[')
      {
        if((c2 == '^' || c2 == 'v' || c2 == 'm' || c2 == 'w') && i % 2 != 0) continue;
        if((c2 == '<' || c2 == '>' || c2 == '[' || c2 == ']') && i % 2 != 1) continue;
        addToConnectorMap(map, x2, y2, pos.x, pos.y, result);
      }
      else if(c2 == 'i' || c2 == 'I' || c2 == 'j' || c2 == 'o' || c2 == 'O')
      {
        addToConnectorMap(map, x2, y2, pos.x, pos.y, result);
      }
      else if(c2 == 'b')
      {
        addAndMark(stack2, x2, y2, pos.x, pos.y, done0, done1, in, w, h);
      }
    }
  }
  return result;
}

/*
parse a diode part
*/
Part* parseDiode(std::vector<Connector>& stack, ConnectorMap& map,
                const Pos& pos,
                std::vector<bool>& done0, std::vector<bool>& done1,
                const std::vector<char>& in, size_t w, size_t h)
{
  (void)stack;
  (void)done0;
  (void)done1;
  Diode* result = new Diode();
  size_t index = pos.y * w + pos.x;
  char c = in[index];
  for(int j = 0; j < 2; j++)
  {
    int i = j * 2;
    if(c == '<' || c == '>') i++;
    
    int x2 = pos.x, y2 = pos.y;
    if(!innerIndexToPos(x2, y2, i, w, h)) continue;
    
    size_t index2 = y2 * w + x2;
    char c2 = in[index2];
    if(c2 == '*' || c2 == 'i' || c2 == 'I' || c2 == 'j' || c2 == 'o' || c2 == 'O'
    || c2 == 'r' || c2 == 'c' || c2 == 'd' || c2 == 'b' || c2 == '+')
    {
      addToConnectorMap(map, x2, y2, pos.x, pos.y, result);
    }
    else if(c2 == '>' || c2 == '<' || c2 == '^' || c2 == 'v'
         || c2 == ']' || c2 == '[' || c2 == 'm' || c2 == 'w')
    {
        if((c2 == '^' || c2 == 'v' || c2 == 'm' || c2 == 'w') && i % 2 != 0) continue;
        if((c2 == '<' || c2 == '>' || c2 == '[' || c2 == ']') && i % 2 != 1) continue;
        addToConnectorMap(map, x2, y2, pos.x, pos.y, result);
    }
  }
  
  if(c == '^') result->outputConnector = Pos(pos.x, pos.y, 0);
  else if(c == '>') result->outputConnector = Pos(pos.x + 1, pos.y, 1);
  else if(c == 'v') result->outputConnector = Pos(pos.x, pos.y + 1, 0);
  else if(c == '<') result->outputConnector = Pos(pos.x, pos.y, 1);
  result->pos = pos;
  
  return result;
}

Part* parseInvertor(std::vector<Connector>& stack, ConnectorMap& map,
                    const Pos& pos,
                    std::vector<bool>& done0, std::vector<bool>& done1,
                    const std::vector<char>& in, size_t w, size_t h)
{
  (void)stack;
  (void)done0;
  (void)done1;
  Invertor* result = new Invertor();
  size_t index = pos.y * w + pos.x;
  char c = in[index];
  for(int j = 0; j < 2; j++)
  {
    int i = j * 2;
    if(c == '[' || c == ']') i++;
    
    int x2 = pos.x, y2 = pos.y;
    if(!innerIndexToPos(x2, y2, i, w, h)) continue;
    
    size_t index2 = y2 * w + x2;
    char c2 = in[index2];
    if(c2 == '*' || c2 == 'i' || c2 == 'I' || c2 == 'j' || c2 == 'o' || c2 == 'O'
    || c2 == 'r' || c2 == 'c' || c2 == 'd' || c2 == 'b' || c2 == '+')
    {
      addToConnectorMap(map, x2, y2, pos.x, pos.y, result);
    }
    else if(c2 == '>' || c2 == '<' || c2 == '^' || c2 == 'v'
         || c2 == ']' || c2 == '[' || c2 == 'm' || c2 == 'w')
    {
        if((c2 == '^' || c2 == 'v' || c2 == 'm' || c2 == 'w') && i % 2 != 0) continue;
        if((c2 == '<' || c2 == '>' || c2 == '[' || c2 == ']') && i % 2 != 1) continue;
        addToConnectorMap(map, x2, y2, pos.x, pos.y, result);
    }
  }
  
  if(c == 'm') result->outputConnector = Pos(pos.x, pos.y, 0);
  else if(c == ']') result->outputConnector = Pos(pos.x + 1, pos.y, 1);
  else if(c == 'w') result->outputConnector = Pos(pos.x, pos.y + 1, 0);
  else if(c == '[') result->outputConnector = Pos(pos.x, pos.y, 1);
  result->pos = pos;
  
  return result;
}


/*
Parse input or output.
*/
Part* parseIO(std::vector<Connector>& stack, ConnectorMap& map,
              const Pos& start,
              std::vector<bool>& done0, std::vector<bool>& done1,
              const std::vector<char>& in, size_t w, size_t h)
{
  (void)stack;
  std::vector<Pos> stack2;
  addAndMark(stack2, start.x, start.y, start.d, done0, done1, w, h);
  char c = in[start.y * w + start.x];
  
  IO* result = (c == 'i' || c == 'I' || c == 'j') ? (IO*)new Input() : (IO*)new Output();
  if(c == 'j') ((Input*)result)->button = true;
  result->num = -1;
  result->state = result->nextstate = false;
  result->highnumbered = false;
  while(!stack2.empty())
  {
    Pos pos = stack2.back();
    stack2.pop_back();
    size_t index = pos.y * w + pos.x;
    char c = in[index];
    result->positions.push_back(pos);
    
    if(c == 'I' || c == 'O')
    {
      result->highnumbered = true;
      result->state = result->nextstate = true;
    }

    for(int i = 0; i < 4; i++)
    {
      int x2 = pos.x, y2 = pos.y;
      if(!innerIndexToPos(x2, y2, i, w, h)) continue;
      
      size_t index2 = y2 * w + x2;
      char c2 = in[index2];
      if(c2 == c || (c == 'i' && c2 == 'I') || (c == 'I' && c2 == 'i')
                 || (c == 'o' && c2 == 'O') || (c == 'O' && c2 == 'o'))
      {
        addAndMark(stack2, x2, y2, pos.x, pos.y, done0, done1, in, w, h);
      }
      else if(c2 == 'i' || c2 == 'I' || c2 == 'j' || c2 == 'o' || c2 == 'O')
      {
        addToConnectorMap(map, x2, y2, pos.x, pos.y, result);
      }
      else if(c2 == '*' || c2 == '+' || c2 == 'b')
      {
        addToConnectorMap(map, x2, y2, pos.x, pos.y, result);
      }
      else if(c2 >= '0' && c2 <= '9')
      {
        result->num = c2 - '0';
        if((!result->highnumbered && result->num < 8) || (result->highnumbered && result->num < 3))
        {
          result->numbered = true;
          result->state = result->nextstate = false;
        }
      }
      else if(c2 == '>' || c2 == '<' || c2 == '^' || c2 == 'v'
           || c2 == ']' || c2 == '[' || c2 == 'm' || c2 == 'w')
      {
        if((c2 == '^' || c2 == 'v' || c2 == 'm' || c2 == 'w') && i % 2 != 0) continue;
        if((c2 == '<' || c2 == '>' || c2 == '[' || c2 == ']') && i % 2 != 1) continue;
        addToConnectorMap(map, x2, y2, pos.x, pos.y, result);
      }
    }
  }
  
  Input* input = dynamic_cast<Input*>(result);
  if(input) input->calcBB();
  return result;
}

/*
parse a xor wire part
*/
Part* parseXorWire(std::vector<Connector>& stack, ConnectorMap& map,
                   const Pos& start,
                   std::vector<bool>& done0, std::vector<bool>& done1,
                   const std::vector<char>& in, size_t w, size_t h)
{
  (void)stack;
  std::vector<Pos> stack2;
  addAndMark(stack2, start.x, start.y, start.d, done0, done1, w, h);
  XorWire* result = new XorWire();
  while(!stack2.empty())
  {
    Pos pos = stack2.back();
    stack2.pop_back();

    for(int i = 0; i < 4; i++)
    {
      int x2 = pos.x, y2 = pos.y;
      if(!innerIndexToPos(x2, y2, i, w, h)) continue;
      
      size_t index2 = y2 * w + x2;
      char c2 = in[index2];
      
      if(c2 == 'r')
      {
        addAndMark(stack2, x2, y2, pos.x, pos.y, done0, done1, in, w, h);
      }
      else if(c2 == '^' || c2 == '>' || c2 == 'v' || c2 == '<'
           || c2 == 'm' || c2 == ']' || c2 == 'w' || c2 == '[')
      {
        if((c2 == '^' || c2 == 'v' || c2 == 'm' || c2 == 'w') && i % 2 != 0) continue;
        if((c2 == '<' || c2 == '>' || c2 == '[' || c2 == ']') && i % 2 != 1) continue;
        addToConnectorMap(map, x2, y2, pos.x, pos.y, result);
      }
    }
  }

  return result;
}

Part* parseCounterWire(std::vector<Connector>& stack, ConnectorMap& map,
                       const Pos& start,
                       std::vector<bool>& done0, std::vector<bool>& done1,
                       const std::vector<char>& in, size_t w, size_t h)
{
  (void)stack;
  std::vector<Pos> stack2;
  addAndMark(stack2, start.x, start.y, start.d, done0, done1, w, h);
  CounterWire* result = new CounterWire();
  while(!stack2.empty())
  {
    Pos pos = stack2.back();
    stack2.pop_back();

    for(int i = 0; i < 4; i++)
    {
      int x2 = pos.x, y2 = pos.y;
      if(!innerIndexToPos(x2, y2, i, w, h)) continue;
      
      size_t index2 = y2 * w + x2;
      char c2 = in[index2];
      
      if(c2 == 'c')
      {
        addAndMark(stack2, x2, y2, pos.x, pos.y, done0, done1, in, w, h);
      }
      else if(c2 == '^' || c2 == '>' || c2 == 'v' || c2 == '<'
           || c2 == 'm' || c2 == ']' || c2 == 'w' || c2 == '[')
      {
        if((c2 == '^' || c2 == 'v' || c2 == 'm' || c2 == 'w') && i % 2 != 0) continue;
        if((c2 == '<' || c2 == '>' || c2 == '[' || c2 == ']') && i % 2 != 1) continue;
        addToConnectorMap(map, x2, y2, pos.x, pos.y, result);
      }
    }
  }

  return result;
}

Part* parseDelayWire(std::vector<Connector>& stack, ConnectorMap& map,
                     const Pos& start,
                     std::vector<bool>& done0, std::vector<bool>& done1,
                     const std::vector<char>& in, size_t w, size_t h)
{
  (void)stack;
  std::vector<Pos> stack2;
  addAndMark(stack2, start.x, start.y, start.d, done0, done1, w, h);
  DelayWire* result = new DelayWire();
  size_t n = 1;
  while(!stack2.empty())
  {
    Pos pos = stack2.back();
    stack2.pop_back();

    for(int i = 0; i < 4; i++)
    {
      int x2 = pos.x, y2 = pos.y;
      if(!innerIndexToPos(x2, y2, i, w, h)) continue;
      
      size_t index2 = y2 * w + x2;
      char c2 = in[index2];
      
      if(c2 == 'd')
      {
        if(!cellDone(x2, y2, pos.x, pos.y, done0, done1, in, w, h)) n++;
        addAndMark(stack2, x2, y2, pos.x, pos.y, done0, done1, in, w, h);
      }
      else if(c2 == '^' || c2 == '>' || c2 == 'v' || c2 == '<'
           || c2 == 'm' || c2 == ']' || c2 == 'w' || c2 == '[')
      {
        if((c2 == '^' || c2 == 'v' || c2 == 'm' || c2 == 'w') && i % 2 != 0) continue;
        if((c2 == '<' || c2 == '>' || c2 == '[' || c2 == ']') && i % 2 != 1) continue;
        addToConnectorMap(map, x2, y2, pos.x, pos.y, result);
      }
    }
  }

  if(n > 30) n = 30;
  result->n = (1 << n) - 2;
  result->tick = result->n; //ensure it's not initially on.
  return result;
}


/*
Parse a group of touching parts.
*/
void parseGroup(std::vector<Part*>& parts, ConnectorMap& map,
                Pos start,
                std::vector<bool>& done0, std::vector<bool>& done1,
                const std::vector<char>& in, size_t w, size_t h)
{
  std::vector<Connector> stack;
  stack.push_back(Connector(start, 0));
  while(!stack.empty())
  {
    Connector co = stack.back();
    Pos pos = co.pos;
    stack.pop_back();
    size_t index = pos.y * w + pos.x;
    char c = in[index];
    if(pos.d == 1) assert(c == 'x' || c == '+');
    
    //d == 0 corresponds to the \ part of x, and the | part of +
    //d == 1 corresponds to the / part of x, and the - part of +

    if(!cellDone(pos.x, pos.y, done0, done1, in, w, h))
    {
      if(c == '*' || c == '+' || c == 'x' || c == 'b')
      {
        parts.push_back(parseWire(stack, map, pos, done0, done1, in, w, h));
      }
      else if(c == '^' || c == '>' || c == 'v' || c == '<')
      {
        parts.push_back(parseDiode(stack, map, pos, done0, done1, in, w, h));
      }
      else if(c == 'm' || c == ']' || c == 'w' || c == '[')
      {
        parts.push_back(parseInvertor(stack, map, pos, done0, done1, in, w, h));
      }
      else if(c == 'r')
      {
        parts.push_back(parseXorWire(stack, map, pos, done0, done1, in, w, h));
      }
      else if(c == 'c')
      {
        parts.push_back(parseCounterWire(stack, map, pos, done0, done1, in, w, h));
      }
      else if(c == 'd')
      {
        parts.push_back(parseDelayWire(stack, map, pos, done0, done1, in, w, h));
      }
      else if(c == 'i' || c == 'I'  || c == 'j' || c == 'o'|| c == 'O')
      {
        parts.push_back(parseIO(stack, map, pos, done0, done1, in, w, h));
      }
    }
  }
}

void parse(std::vector<Part*>& parts, const std::vector<char>& in, size_t w, size_t h)
{
  size_t x = 0, y = 0, d = 0;
  std::vector<bool> done0(w * h, false);
  std::vector<bool> done1(w * h, false);
  ConnectorMap map;
  for(;;)
  {
    if(y >= h || (y == h && x >= w)) break;
    size_t index = y * w + x;
    if(index >= in.size()) break;
    char c = in[index];
    bool notdone = !cellDone(x, y, done0, done1, in, w, h);
    bool secondok = (d == 0 || c == 'x' || c == '+');


    if(notdone && secondok && (c == '*' || c == '+' || c == 'x'
    || (c >= '0' && c  <= '9')
    || c == '<' || c == '>' || c == '^' || c == 'v' || c == '[' || c == ']'
    || c == 'm' || c == 'w' || c == 'r' || c == 'c' || c == 'd' || c == 'b'
    || c == 'i' || c == 'I'|| c == 'j'|| c == 'o'|| c == 'O'))
    {
      parseGroup(parts, map, Pos(x, y, !done0[index] ? 0 : 1), done0, done1, in, w, h);
    }

    d++;
    if(d > 1)
    {
      x++;
      if(x >= w)
      {
        x = 0;
        y++;
      }
      d = 0;
    }
  }
  resolveConnectors(map);
}
