/*
LLL Programming Language Interpreter

Copyright (c) 2012 Lode Vandevenne
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
  * Neither the name of Lode Vandevenne nor the names of his contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#pragma once

#include <vector>
#include <string>
#include <map>
#include <iostream>

//Remove this line to work in console mode only, but not require linking to SDL.
#define SDLGRAPHICS

extern bool interactive; //whether we're in interactive mode (-i command line, only used if SDLGRAPHICS compile is enabled)

#ifdef SDLGRAPHICS
#include "quickcg.h"
using namespace QuickCG;

//Bad! Global variables! I know!
extern bool mouseJustDownCircuit;
extern bool mouseJustDown;
extern int mouseX;
extern int mouseY;
extern bool LMB;
extern bool RMB;

extern int pixelsize;
#endif //SDLGRAPHICS

/*
+--A--+--B--+--C--+-----+
|     |     |     |     |
D 0,0 E 1,0 F 2,0 | 3,0 |
|     |     |     |     |
+--G--+-----+-----+-----+
|     |     |     |     |
H 0,1 | 1,1 | 2,1 | 3,1 |
|     |     |     |     |
+-----+-----+-----+-----+
|     |     |     |     |
| 0,2 | 1,2 | 2,2 | 3,2 |
|     |     |     |     |
+-----+-----+-----+-----+

How the 'd' parameter works for connection-coordinates.
Note: Sometimes the d is used as described here, at other times it is used to
distinguish between the two parts of a '+' or 'x'.

A: x=0, y=0, d=h --> 0,0,0
B: x=1, y=0, d=h --> 1,0,0
C: x=2, y=0, d=h --> 2,0,0
D: x=0, y=0, d=v --> 0,0,1
E: x=1, y=0, d=v --> 1,0,1
F: x=2, y=0, d=v --> 2,0,1
G: x=0, y=1, d=h --> 0,1,0
H: x=0, y=1, d=v --> 0,1,1


e.g.:
connector 2,5,0 connects cell 1,5 and 2,5
connector 2,5,1 connects cell 2,4 and 2,5
connector A,B,0 connects cell A-1,B and A,B
connector A,B,1 connects cell A,B-1 and A,B


        x,y,0
      +-------+
      |       |
x,y,1 |  x,y  | x+1,y,1
      |       |
      +-------+
       x,y+1,0

*/


//Input/Output with the host computer
struct ComputerIO
{
  unsigned char in; //8 bits
  bool in_update;
  bool in_eof;
  
  std::string inputbuffer;
  
  unsigned char out; //8 bits
  bool out_update;
  bool out_prev_update;
  bool out_inputready; //indicates that program input is ready to receive
  bool out_eof; //means: exit program
};

extern ComputerIO io;

/*
Coordinates.
d is used in case of connector coordinates, see diagram above.
d is also used to indicate the part of '+' and 'x' wire connections
*/
struct Pos
{
  int x; //x coordinate of line between two cells
  int y; //y coordinate of line between two cells
  int d; //dir, 0 = two cells above each other, 1 = two cells next to each other

  Pos() {}
  Pos(int x, int y) : x(x), y(y), d(0) {}
  Pos(int x, int y, int d) : x(x), y(y), d(d) {}
  
  bool operator==(const Pos& other) const
  {
    return x == other.x && y == other.y && d == other.d;
  }
};
  
inline bool operator<(const Pos& a, const Pos& b)
{
  if(a.x == b.x)
  {
    if(a.y == b.y) return a.d < b.d;
    return a.y < b.y;
  }
  return a.x < b.x;
}

class Part;

struct Connector
{
  Connector() {}
  Connector(Pos pos, Part* other) : pos(pos), other(other) {}
  Connector(int x, int y, int d, Part* other) : pos(Pos(x, y, d)), other(other) {}
  Pos pos; //connector coordinates
  Part* other;
};

template<typename T>
void cleanPointerVector(T& v)
{
  for(size_t i = 0; i < v.size(); i++) delete v[i];
}

//used in std::map as value to represent "Part[2]" as type.
class TwoParts
{
  Part* part0;
  Part* part1;
  
  public:
  
  Part*& operator[](int i)
  {
    return i == 0 ? part0 : part1;
  }
};

/*
if pos.d == 0, then topmost part is part[0], bottom part is part[1]
if pos.d == 1, then left part is part[0], right part is part[1]
*/
typedef std::map<Pos, TwoParts> ConnectorMap;

