/*
LLL Programming Language Interpreter

Copyright (c) 2012 Lode Vandevenne
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
  * Neither the name of Lode Vandevenne nor the names of his contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*
Compile command:
g++ src/*.cpp -lSDL -O3 -o lll

Run command, e.g.:
./lll examples/div.txt

./lll -c examples/hello.txt
*/

#include "main.h"

#include <cassert>
#include <iostream>
#include <fstream>
#include <cassert>
#include <cstdlib>

#include "parse.h"
#include "parts.h"

#ifdef SDLGRAPHICS
//Bad! Global variables! I know!
bool mouseJustDownCircuit = false;
bool lastMouseCircuit = false;
bool mouseJustDown = false;
bool lastMouse = false;
int mouseX = 0;
int mouseY = 0;
bool LMB = false;
bool RMB = false;
int pixelsize = 32;
std::string sdloutstring;

static const size_t SCREENW = 1024;
static const size_t SCREENH = 768;

bool interactive = true;
#else
bool interactive = false;
#endif

double sleepDelay = 0.1;
int tickMode = 1; //0: paused, 1: use sleep delay, 2: 1 tick per frame, 3: many ticks per frame (25FPS)
ComputerIO io;


#ifndef SDLGRAPHICS //because these functions are also in QuickCG, which is SDL-only.
void loadFile(std::vector<unsigned char>& buffer, const std::string& filename) //designed for loading files from hard disk in an std::vector
{
  std::ifstream file(filename.c_str(), std::ios::in|std::ios::binary|std::ios::ate);

  //get filesize
  std::streamsize size = 0;
  if(file.seekg(0, std::ios::end).good()) size = file.tellg();
  if(file.seekg(0, std::ios::beg).good()) size -= file.tellg();

  //read contents of the file into the vector
  buffer.resize(size_t(size));
  if(size > 0) file.read((char*)(&buffer[0]), size);
}

//write given buffer to the file, overwriting the file, it doesn't append to it.
void saveFile(const std::vector<unsigned char>& buffer, const std::string& filename)
{
  std::ofstream file(filename.c_str(), std::ios::out|std::ios::binary);
  file.write(buffer.size() ? (char*)&buffer[0] : 0, std::streamsize(buffer.size()));
}
#endif //no SDLGRAPHICS

//Turn the source text file in the 2D grid.
void readSource(size_t& w, size_t& h, std::vector<char>& out, const std::vector<unsigned char>& in)
{
  size_t width = 0, height = 1;
  size_t currentw = 0;
  for(size_t i = 0; i < in.size(); i++)
  {
    char c = in[i];
    if(c == 10)
    {
      height++;
      if(currentw > width) width = currentw;
      currentw = 0;
    }
    else if(c == 13) ; //Ignore.
    else currentw++;
  }

  out.resize(width * height);

  bool comment = false;
  size_t x = 0, y = 0;
  for(size_t i = 0; i < in.size(); i++)
  {
    char c = in[i];
    if(c == 10)
    {
      for(size_t j = x; j < width; j++) out[y * width + j] = ' ';
      y++;
      x = 0;
    }
    else if(c == 13) ; //Ignore.
    else
    {
      if(c == '"') comment = !comment;
      out[y * width + x] = comment ? ' ' : c;
      x++;
    }
  }
  
  w = width;
  h = height;
}


  
  
#ifdef SDLGRAPHICS
//The color of each cell.
ColorRGB charToColor(char c)
{
  switch(c)
  {
    case '*': return ColorRGB(128, 0, 0);
    case '+': return ColorRGB(160, 0, 0);
    case 'x': return ColorRGB(170, 0, 0);
    case '^': return ColorRGB(192, 96, 0);
    case '>': return ColorRGB(192, 97, 0);
    case 'v': return ColorRGB(192, 98, 0);
    case '<': return ColorRGB(192, 99, 0);
    case 'm': return ColorRGB(192, 0, 0);
    case ']': return ColorRGB(192, 1, 0);
    case 'w': return ColorRGB(192, 2, 0);
    case '[': return ColorRGB(192, 3, 0);
    case 'b': return ColorRGB(255, 255, 0);
    case 'r': return ColorRGB(255, 128, 255);
    case 'c': return ColorRGB(128, 255, 255);
    case 'd': return ColorRGB(192, 192, 128);
    case 'i': return ColorRGB(128, 128, 128);
    case 'I': return ColorRGB(128, 128, 129);
    case 'j': return ColorRGB(128, 128, 96);
    case 'o': return ColorRGB(0, 96, 0);
    case 'O': return ColorRGB(0, 97, 0);
    case '0': return ColorRGB(0, 0, 80);
    case '1': return ColorRGB(1, 0, 80);
    case '2': return ColorRGB(2, 0, 80);
    case '3': return ColorRGB(3, 0, 80);
    case '4': return ColorRGB(4, 0, 80);
    case '5': return ColorRGB(5, 0, 80);
    case '6': return ColorRGB(6, 0, 80);
    case '7': return ColorRGB(7, 0, 80);
    case '8': return ColorRGB(8, 0, 80);
    case '9': return ColorRGB(9, 0, 80);
    case '.': return ColorRGB(10, 0, 0);
    default: return ColorRGB(0, 0, 0);
  }
}

bool mouseClicked(int x0, int y0, int x1, int y1)
{
  return mouseJustDown && mouseX >= x0 && mouseX < x1 && mouseY >= y0 && mouseY < y1;
}

//Handle the "GUI" controls of the SDL version.
void handleControls()
{
  print(sdloutstring, 2, SCREENH - 12);
  //speed control panel
  print("speed: 0, 1, 2, 3, 4, 5, 6", SCREENW - 224, SCREENH - 12);
  if(mouseClicked(SCREENW - 168 - 4, SCREENH - 14, SCREENW - 160 + 4, SCREENH))
  {
    tickMode = 0;
    sleepDelay = 20.0;
  }
  else if(mouseClicked(SCREENW - 168 + 24 - 4, SCREENH - 14, SCREENW - 160 + 24 + 4, SCREENH))
  {
    tickMode = 1;
    sleepDelay = 2.0;
  }
  else if(mouseClicked(SCREENW - 168 + 2 * 24 - 4, SCREENH - 14, SCREENW - 160 + 2 * 24 + 4, SCREENH))
  {
    tickMode = 1;
    sleepDelay = 0.5;
  }
  else if(mouseClicked(SCREENW - 168 + 3 * 24 - 4, SCREENH - 14, SCREENW - 160 + 3 * 24 + 4, SCREENH))
  {
    tickMode = 1;
    sleepDelay = 0.1;
  }
  else if(mouseClicked(SCREENW - 168 + 4 * 24 - 4, SCREENH - 14, SCREENW - 160 + 4 * 24 + 4, SCREENH))
  {
    tickMode = 1;
    sleepDelay = 0.05;
  }
  else if(mouseClicked(SCREENW - 168 + 5 * 24 - 4, SCREENH - 14, SCREENW - 160 + 5 * 24 + 4, SCREENH))
  {
    tickMode = 2;
    sleepDelay = 0.01;
  }
  else if(mouseClicked(SCREENW - 168 + 6 * 24 - 4, SCREENH - 14, SCREENW - 160 + 6 * 24 + 4, SCREENH))
  {
    tickMode = 3;
    sleepDelay = 0.0;
  }
  int speedIndicator = 0;
  if(tickMode == 1)
  {
    if(sleepDelay >= 2.0) speedIndicator = 1;
    else if(sleepDelay >= 0.5) speedIndicator = 2;
    else if(sleepDelay >= 0.1) speedIndicator = 3;
    else speedIndicator = 4;
  }
  else if(tickMode == 2) speedIndicator = 5;
  else if(tickMode == 3) speedIndicator = 6;
  print(valtostr(speedIndicator), SCREENW - 168 + speedIndicator * 24, SCREENH - 12, RGB_Red);
}

void handleGraphics(const std::vector<Part*>& parts, const std::vector<char>& source, size_t w, size_t h)
{
  bool realMouseJustDownCircuit = LMB && !lastMouseCircuit;
  lastMouseCircuit = LMB;
  if(realMouseJustDownCircuit) mouseJustDownCircuit = true;

  std::vector<bool> lightmap(w * h, false);

  for(size_t i = 0; i < parts.size(); i++)
  {
    DirectedStatedPart* d = dynamic_cast<DirectedStatedPart*>(parts[i]);
    if(d && d->state) lightmap[w * d->pos.y + d->pos.x] = true;
    IO* io = dynamic_cast<IO*>(parts[i]);
    if(io && io->state)
    {
      for(size_t i = 0; i < io->positions.size(); i++)
      {
        Pos pos = io->positions[i];
        lightmap[w * pos.y + pos.x] = true;
      }
    }
  }

  for(size_t y = 0; y < h; y++)
  for(size_t x = 0; x < w; x++)
  {
    if((x + 1) * pixelsize > SCREENW || (y + 1) * pixelsize > SCREENH) continue;
    
    ColorRGB c = charToColor(source[y * w + x]);
    
    if(lightmap[w * y + x])
    {
      c.r = (int)(c.r * 1.5); if(c.r > 255) c.r = 255;
      c.g = (int)(c.g * 1.5); if(c.g > 255) c.g = 255;
      c.b = (int)(c.b * 1.5); if(c.b > 255) c.b = 255;
      char s = source[y * w + x];
      if(s == 'o' || s == 'O' || s == 'i' || s == 'I' || s == 'j')
      {
        c.r = (int)(c.r * 1.5 + 32); if(c.r > 255) c.r = 255;
        c.g = (int)(c.g * 1.5 + 32); if(c.g > 255) c.g = 255;
        c.b = (int)(c.b * 1.5 + 32); if(c.b > 255) c.b = 255;
      }
    }
    
    for(int x2 = 0; x2 < pixelsize; x2++)
    for(int y2 = 0; y2 < pixelsize; y2++)
    {
      size_t x3 = x * pixelsize + x2;
      size_t y3 = y * pixelsize + y2;
      
      pset(x3, y3, c);
    }
  }
  
  redraw();
  if(sleepDelay >= 0.05) SDL_Delay(5); //consumes too much CPU, even with done()'s delay
  cls();
}
#endif //SDLGRAPHICS


//Handle PC<->program communication, both for console and interactive mode.
void handleIO()
{
  if(!io.out_prev_update && io.out_update)
  { 
    if(interactive)
    {
#ifdef SDLGRAPHICS
      sdloutstring.push_back(io.out);
      if(sdloutstring.size() > 32) sdloutstring = sdloutstring.substr(1, sdloutstring.size());
#endif
    }
    else
    {
      std::cout << (char)(io.out) << std::flush;
    }
  }
  
  if(!interactive && !io.in_eof && io.out_inputready)
  {
    if(std::cin.eof()) io.in_eof = true;
    
    if(io.in_update || io.in_eof)
    {
      io.in_update = false;
    }
    else
    {
      io.in_update = true;
      
      if(io.inputbuffer.size() == 0)
      {
        std::cout << std::endl << "enter input: ";
        std::getline(std::cin, io.inputbuffer);
        while(io.inputbuffer.empty())
        {
          if(std::cin.eof())
          {
            io.in_eof = true;
            std::cout << "<EOF>" << std::endl;
            break;
          }
          std::cout << std::endl << "error, please enter 1 or more characters, or press CTRL+C to quit: ";
          std::getline(std::cin, io.inputbuffer);
          if(io.inputbuffer == "ragequit") std::exit(0);
        }
      }
      
      if(io.inputbuffer.size() > 0)
      {
        io.in = io.inputbuffer[0];
        io.inputbuffer = io.inputbuffer.substr(1, io.inputbuffer.size());
      }
    }
  }
  else if(interactive)
  {
#ifdef SDLGRAPHICS
    unsigned char c = getInputCharacter();
    if(c)
    {
      io.in = c;
      io.in_update = false;
    }
    else io.in_update = true;
#endif
  }

  io.out = 0;
  io.out_prev_update = io.out_update;
  io.out_update = false;
}

////////////////////////////////////////////////////////////////////////////////


int main(int argc, char* argv[])
{
  
  if(argc < 2)
  {
    std::cout << "please provide a filename" << std::endl;
    return 1;
  }
  
  
  std::string filename = argv[1];
  
  if(filename == "-i" || filename == "-c")
  {
    if(argc < 3)
    {
      std::cout << "please provide a filename" << std::endl;
      return 1;
    }
    if(filename == "-i")
    {
#ifdef SDLGRAPHICS
      interactive = true;
#else
      std::cout << "Interactive mode not supported, because it was compiled without SDL graphics." << std::endl;
#endif
    }
    else
    {
      interactive = false;
    }
    filename = argv[2];
  }
  else if(filename == "-h")
  {
    std::cout << "LLL interpreter by Lode Vandevenne (2012).\n"
              << "Usage: " << std::string(argv[0]) << " [OPTION]... FILE\n"
              << "  -c: run in console mode\n"
              << "  -h: show this help\n"
              << "  -i: run in interactive mode (requires SDL)\n";
    return 0;
  }

  std::vector<unsigned char> file;
  loadFile(file, filename);
  if(file.size() == 0)
  {
    std::cout << "File empty or unexistant" << std::endl;
    return 0;
  }
  
  size_t w, h;
  std::vector<char> source;
  readSource(w, h, source, file);
  
  std::cout << "Welcome to LLL. " << "w: " << w << " h: " << h << std::endl;
  
  std::vector<Part*> parts;
  parse(parts, source, w, h);
  
  
#ifdef SDLGRAPHICS
  
  if(interactive)
  {
    screen(SCREENW, SCREENH, 0, "LLL by Lode Vandevenne");
    
    pixelsize = 32;
    while(pixelsize > 1 && (w * pixelsize > SCREENW  || h * pixelsize > SCREENH))
    {
      pixelsize--;
    }
  }
  
  double lastUpdate = getTime();
  
  
#endif //SDLGRAPHICS
  io.in_eof = false;

  //warmup with avoid blinking NOTs and unstable initial AND gates.
  for(size_t i = 0; i < parts.size(); i++)
  {
    parts[i]->initialize();
  }
  /*for(size_t i = 0; i < parts.size(); i++)
  {
    parts[i]->update();
  }*/
  initphase = false;
  std::cout << "Press CTRL+C to quit." << std::endl;

  for(;;)
  {
#ifdef SDLGRAPHICS
    if(interactive && done()) break;
    
    getMouseState(mouseX, mouseY, LMB, RMB);
    mouseJustDown = LMB && !lastMouse;
    lastMouse = LMB;
    
#endif //SDLGRAPHICS
    //gives too much non-ascii junk characters in terminal in interactive mode
    
    if(!interactive && io.out_eof)
    {
      std::cout << std::endl;
      break; //exit program
    }
    

    int numTicks = 1;

#ifdef SDLGRAPHICS
    double ticksBegin = getTime(); //for tickMode 3
    if(interactive)
    {
      numTicks = 0;
      if(tickMode == 1 && getTime() - lastUpdate > sleepDelay) numTicks = 1;
      if(tickMode == 2) numTicks = 1;
    }
#endif
    for(;;)
    {
      if(tickMode == 3)
      {
#ifdef SDLGRAPHICS
        if(getTime() - ticksBegin > 0.04) break;
#endif
      }
      else
      {
        numTicks--;
        if(numTicks < 0) break;
      }
      
      handleIO();
      
      for(size_t i = 0; i < parts.size(); i++)
      {
        parts[i]->calculate();
      }
      for(size_t i = 0; i < parts.size(); i++)
      {
        parts[i]->update();
      }
#ifdef SDLGRAPHICS
      if(interactive)
      {
        mouseJustDownCircuit = false;
        lastUpdate = getTime();
      }
#endif
    }
  
  
#ifdef SDLGRAPHICS
    if(interactive)
    {
      handleControls();
      handleGraphics(parts, source, w, h);
    }
#endif //SDLGRAPHICS
  }

  cleanPointerVector(parts);
  return 0;
}
