/*
Gammaplex version 0.34 Interpreter version 20100514

Copyright (c) 2004-2007, Lode Vandevenne
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

*) Redistributions of source code must retain the above copyright notice, this 
list of conditions and the following disclaimer.
*) Redistributions in binary form must reproduce the above copyright notice, 
this list of conditions and the following disclaimer in the documentation and/or
other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*
Changelog:

2010: small bugfixes and solved g++ warnings

2007: updated interpreter, used new QuickCG version, and made it more than 10x faster by not handling SDL events each frame

2004: initial version
*/

/*
Note that there still may be some commands that can behave buggy,
because not all commands are tested. If your program does not behave
like the Gammaplex specification states, please let me know. Contact
info is in quickcg.h.
*/

/*
Example of what this interpreter runs:

@   200)u150)l1a0#0}>1a{"}4#0XG  v
Mandelbrot by       ^   v?,y(]0  <
Lode Vandevenne         >1a{s"sN}v
v0.1                ^    E?,h(]1R<
0](y2:-1.5*0.5y*:0.5-30])        \
1](h2:-0.5h*:31])                \
32]0)u0)u0)u0)u0)                \
>36](")34](32])35](33])32](w*33](\
w*-30](+34])2#32](*33](*31](+35])\
34](w*35](w*+4,?v36](16,?v7#0G   \
           G0#11<        >12#0G  \
>255#255#36](16*H3a}PXg           
>128#128#128#3a}PXg               
*/

#include <cmath>
#include <SDL/SDL.h>
#include <iostream>
#include <fstream>
#include <algorithm>
#include <string>
#include "quickcg.h"

const int RA = 1048576;
const int SA = 1048576;

const int MAXIW = 1024;
const int MAXIH = 1024;

const int MAXGOSUB = 1024;

const int MAXSCREENU = 1600;
const int MAXSCREENV = 1200;
Uint32 screenBuffer[MAXSCREENU * MAXSCREENV]; //will contain the pixels

using namespace QuickCG;

Uint8 instruction[MAXIW][MAXIH];

Uint32 ix = 0, iy = 0; //instruction pointer
Uint32 iw, ih; //width and height of actual instruction matrix
Uint32 id = 1; //instruction pointer direction (N, E, S, W); 1 = E (to the right)
double sc[RA]; //max 65536 doubles in memory, the first 8 are parameters of functions
double st[SA]; //a stack to put extra values on
Uint32 rp, sp; //the register pointer, and the stack pointer (stack pointer only used internally)
double dc = 1; //the decimal counter: 1 = will create new number on stack, 0 = enter digits before point, 0.1 - 0.0000000...00001 = enter digits after the point
double anchorTime = 5; //minimum time between two time anchors in milliseconds
double lastAnchorTime; //time at previous time anchor
int instructionMode = 0; //extended or string mode: 0 = normal, 1 = extended command, 2 = string mode, 3 = temporary normal command mode while in string mode
int currentInstruction;
int debugMode = 0;
int gosubx[MAXGOSUB]; //for RETURN commands of GOSUBS
int gosuby[MAXGOSUB];
int gosubd[MAXGOSUB];
int gp; //gosub pointer
Uint32 ticks = 0;
int numStackProgram = 0; //hoeveel registers / stack kopieren met stack program
int stackProgram[16]; //16 register addressen voor te poppen / pushen
int stringCommandLifeSpan = 0;

bool escapeRelease = 0;


const int PROGRAMBUFFER = 1048576;
int startX, startY; //here the ip will start

void loadCodeFile(const char* filename)
{
  //load up the file
  //any character with ascii code <32 is seen as "enter" value!

  std::vector<unsigned char> buffer(PROGRAMBUFFER); //max progarm size currently 1024x1024

  loadFile(buffer, filename);
  if(buffer.empty())
  {
    cls();
    print("File not found or empty. Default code file is gamma.txt, load other files with command line parameter or from within this program with escape, then option 3. Press any key to continue.");
    redraw();
    sleep();
  }

  //parse file
  //line 1: first line of the instructions, and get width
  if(!buffer.empty())
  {
    int filePos = 0;
    int iPosX = 0, iPosY = 0;
    while(buffer[filePos]>=32 && filePos < (int)buffer.size())
    {
      instruction[iPosX][0] = buffer[filePos];
      if(buffer[filePos] == '@') {startX = iPosX; startY = iPosY;}
      filePos++;
      iPosX++;
    }
    iw = iPosX;
    iPosY++;
    iPosX = 0;
  
    while(filePos < (int)buffer.size())
    {
      instruction[iPosX][iPosY] = buffer[filePos];
      if(buffer[filePos] == '@') {startX = iPosX; startY = iPosY;}
      if(buffer[filePos] < 32) {filePos++; continue;}
      filePos++;
      iPosX++;
      if(iPosX >= (int)iw)
      {
        iPosX = 0;
        iPosY++;
      }
    }
    ih = iPosY;

    ix = startX;
    iy = startY;
  }
  else
  {
    iw = ih = 1;
    instruction[0][0] = 0;
    ix = 0; iy = 0; id = 0;
  }
}

void push(double value)
{
  sp++;
  sp%=SA;
  st[sp] = value;
}

double pop()
{
  double value = st[sp];
  sp--;
  sp%=SA;
  return value;
}

inline void changeDC(int digit)
{
  if(dc == 1) 
  {
    push(digit); 
    dc = 0;
  }
  else if(dc == 0) 
  {
    st[sp] *= 10;
    st[sp] += digit;
  }  
  else //dc = 0.1 - 0.000000...000001
  {
    st[sp] += dc * digit;
    dc /= 10.0;
  }  
} 

inline void moveIP()
{
  switch(id)
  {
    case 0: iy--; break;
    case 1: ix++; break;
    case 2: iy++; break;
    case 3: ix--; break;
  };
  ix = (ix + iw) % iw;
  iy = (iy + ih) % ih;
}  

void printDebugInfo()
{
  print("ix:",0,0,ColorRGB(255,255,0),1); print(ix,24,0,ColorRGB(255,255,255),1);
  print("iy:",0,9,ColorRGB(255,255,0),1); print(iy, 24, 8,ColorRGB(255,255,255),1);
  print("id:",0,16,ColorRGB(255,255,0),1); print(id, 24, 16, ColorRGB(255, 255, 255), 1);
  print("instr:",0,24,ColorRGB(255,255,0),1); print(currentInstruction, 48, 24,ColorRGB(255,255,255),1);
  print("im:",64,24,ColorRGB(255,255,0),1); print(instructionMode,88,24,ColorRGB(255,255,255),1);
  print("ticks:",0,32,ColorRGB(255,255,0),1); print(ticks, 48, 32,ColorRGB(255,255,255),1);
  print("dc:",0,40,ColorRGB(255,255,0),1); fprint(dc, 4, 24, 40, ColorRGB(255, 255, 255), 1);
  fprint(sc[0%RA], 4, 0, 48, ColorRGB(255, 0, 0),1);
  fprint(sc[1%RA], 4, 0, 56, ColorRGB(255, 0, 0),1);
  fprint(sc[2%RA], 4, 0, 64, ColorRGB(255, 0, 0),1);
  fprint(sc[3%RA], 4, 0, 72, ColorRGB(255, 0, 0),1);
  fprint(sc[4%RA], 4, 0, 80, ColorRGB(255, 0, 0),1);
  fprint(sc[5%RA], 4, 0, 88, ColorRGB(255, 0, 0),1);
  fprint(sc[6%RA], 4, 0, 96, ColorRGB(255, 0, 0),1);
  fprint(sc[7%RA], 4, 0, 104, ColorRGB(255, 0, 0),1);
  fprint(sc[8%RA], 4, 0, 112, ColorRGB(255, 0, 0),1);
  fprint(sc[9%RA], 4, 0, 120, ColorRGB(255, 0, 0),1);
  fprint(sc[10%RA], 4, 0, 128, ColorRGB(255, 0, 0),1);
  fprint(sc[11%RA], 4, 0, 136, ColorRGB(255, 0, 0),1);
  fprint(sc[12%RA], 4, 0, 144, ColorRGB(255, 0, 0),1);
  fprint(st[(sp-0)%SA], 4, 0, 152, ColorRGB(0, 255, 0),1);
  fprint(st[(sp-1)%SA], 4, 0, 160, ColorRGB(0, 255, 0),1);
  fprint(st[(sp-2)%SA], 4, 0, 168, ColorRGB(0, 255, 0),1);
  fprint(st[(sp-3)%SA], 4, 0, 176, ColorRGB(0, 255, 0),1);
  fprint(st[(sp-4)%SA], 4, 0, 184, ColorRGB(0, 255, 0),1);
  fprint(st[(sp-5)%SA], 4, 0, 192, ColorRGB(0, 255, 0),1);
  fprint(st[(sp-6)%SA], 4, 0, 200, ColorRGB(0, 255, 0),1);
  fprint(st[(sp-7)%SA], 4, 0, 208, ColorRGB(0, 255, 0),1);
  fprint(st[(sp-8)%SA], 4, 0, 216, ColorRGB(0, 255, 0),1);
  fprint(st[(sp-9)%SA], 4, 0, 224, ColorRGB(0, 255, 0),1);
  fprint(st[(sp-10)%SA], 4, 0, 232, ColorRGB(0, 255, 0),1);
  fprint(st[(sp-11)%SA], 4, 0, 240, ColorRGB(0, 255, 0),1);
}

inline void moveIPBack()
{
  switch(id)
  {
    case 0: iy++; break;
    case 1: ix--; break;
    case 2: iy--; break;
    case 3: ix++; break;
  };
  ix = (ix + iw) % iw;
  iy = (iy + ih) % ih;
}

//GAMMADRAW

void GgetRGB(Uint32 c, Uint8& R, Uint8& G, Uint8& B)
{
  R = (c / 65536) % 256;
  G = (c / 256) % 256;
  B = c % 256;
} 

void GsetRGB(Uint32& c, Uint8 R, Uint8 G, Uint8 B)
{
  c = B + 256 * G + 65536 * R;
}

void Gpset(int x, int y, bool textBG = 0)
{
  x = abs(x) % w;
  y = abs(y) % h;
  
  int style = abs(int(sc[12]));
  Uint8 r = Uint8(sc[2]);
  Uint8 g = Uint8(sc[3]);
  Uint8 b = Uint8(sc[4]);
    
  //skip the rest if no style
  if(style == 0) 
  {
    GsetRGB(screenBuffer[h * x + y], r, g, b);
    return;
  }  
  Uint8 r2 = Uint8(sc[8]);
  Uint8 g2 = Uint8(sc[9]);
  Uint8 b2 = Uint8(sc[10]);
  
  //pure text bg
  if(style == 4096)
  {
    GsetRGB(screenBuffer[h * x + y], r2, g2, b2);
    return; 
  }  
  
  Uint8 T = Uint8(sc[11]);
  Uint8 r3, g3, b3;
  GgetRGB(screenBuffer[h * x + y], r3, g3, b3);
  Uint8 R, G, B; //the final R, G and B
  
  //text background color: swap color1 and color2
  if(textBG)
  {
    r += r2; r2 = r - r2; r -= r2;
    g += g2; g2 = g - g2; g -= g2;
    b += b2; b2 = b - b2; b -= b2;
  }  
  
  //make underlying pixels negative (style &4)
  if(style & 4)
  {
    r3 = 255 - r3;
    g3 = 255 - g3;
    b3 = 255 - b3;
  }  

  //gradients 
  if((style & 1024) && !(style & 2048)) 
  {
    Uint8 xmod = 255 - Uint8(256.0 * (w - x) / double(w));
    r = (r * (255 - xmod) + r2 * xmod) / 256;
    g = (g * (255 - xmod) + g2 * xmod) / 256;
    b = (b * (255 - xmod) + b2 * xmod) / 256;
  }  
  if((style & 2048) && !(style & 1024))
  {
    Uint8 ymod = 255 - Uint8(256.0 * (w - y) / double(w));
    r = (r * (255 - ymod) + r2 * ymod) / 256;
    g = (g * (255 - ymod) + g2 * ymod) / 256;
    b = (b * (255 - ymod) + b2 * ymod) / 256;
  } 
  if((style & 1024) && (style & 2048))
  {
    Uint8 mod = 255 - Uint8(256.0 * (w + h - x - y) / double(w + h));
    r = (r * (255 - mod) + r2 * mod) / 256;
    g = (g * (255 - mod) + g2 * mod) / 256;
    b = (b * (255 - mod) + b2 * mod) / 256;
  }
  R = r; G = g; B = b; //has to be done after gradient calculations
  
  //transparency and blending styles
  if((style & 1) && !(style & 2)) //transparency
  {
    R = (r * (256 - T) + r3 * T) / 256;
    G = (g * (256 - T) + g3 * T) / 256;
    B = (b * (256 - T) + b3 * T) / 256;
  }  
  if((style & 2) && !(style & 1)) //multiply blending
  {
    R = (r * r3) / 256;
    G = (g * g3) / 256;
    B = (b * b3) / 256;
    //do transparency after multiply blending
    R = (R * (255 - T) + r3 * T) / 256;
    G = (G * (255 - T) + g3 * T) / 256;
    B = (B * (255 - T) + b3 * T) / 256;
  }
  if((style & 2) && (style & 1)) //mysterious blending (darken / lighten)
  {
    int rmod = (r - 128) * 2;
    R = std::max(0, std::min(255, r3 + rmod));
    int gmod = (g - 128) * 2;
    G = std::max(0, std::min(255, g3 + gmod));
    int bmod = (b - 128) * 2;
    B = std::max(0, std::min(255, b3 + bmod));
    //do transparency after mysterious blending
    R = (R * (255 - T) + r3 * T) / 256;
    G = (G * (255 - T) + g3 * T) / 256;
    B = (B * (255 - T) + b3 * T) / 256;
  }
  
  //do and dont draw   
  if((style & 8) && !(style & 16)) 
  {
    if(r2 == r3 && g2 == g3 && b2 == b3)
    R = r3;
    G = g3;
    B = b3;
  }  
  if((style & 16) && !(style & 8))
  {
    if(!(r2 == r3 && g2 == g3 && b2 == b3))
    R = r3;
    G = g3;
    B = b3;
  } 
  if((style & 8) && (style & 16))
  {
    if(r2 == r3 && g2 == g3 && b2 == b3)
    R = (R + r3) / 2;
    G = (G + g3) / 2;
    B = (B + b3) / 2;
  }
  
  //dot and line patterns
  bool invertPattern = style & 512;
  if(style & 32)
  {
    if(((x + y) % 2 == 0) ^ invertPattern)
    {
      R = r3;
      G = g3;
      B = b3;
    }  
  }  
  if(style & 64)
  {
    if((x % 3 == 0 && y % 3 == 0) ^ invertPattern)
    {
      R = r3;
      G = g3;
      B = b3;
    }  
  }  
  if(style & 128)
  {
    if((y % 2 == 0) ^ invertPattern)
    {
      R = r3;
      G = g3;
      B = b3;
    }  
  }  
  if(style & 256)
  {
    if((x % 2 == 0) ^ invertPattern)
    {
      R = r3;
      G = g3;
      B = b3;
    }
  }
  
  GsetRGB(screenBuffer[h * x + y], R, G, B);
} 

bool Gline(int x1, int y1, int x2, int y2)
{
  if(x1 < 0 || x1 > w - 1 || x2 < 0 || x2 > w - 1 || y1 < 0 || y1 > h - 1 || y2 < 0 || y2 > h - 1) return 0;
  
  int deltax = abs(x2 - x1); //The difference between the x's
  int deltay = abs(y2 - y1); //The difference between the y's
  int x = x1; //Start x off at the first pixel
  int y = y1; //Start y off at the first pixel
  int xinc1, xinc2, yinc1, yinc2, den, num, numadd, numpixels, curpixel;

  if (x2 >= x1) //The x-values are increasing
  {
    xinc1 = 1;
    xinc2 = 1;
  }
  else //The x-values are decreasing
  {
    xinc1 = -1;
    xinc2 = -1;
  }
  if (y2 >= y1) //The y-values are increasing
  {
    yinc1 = 1;
    yinc2 = 1;
  }
  else //The y-values are decreasing
  {
    yinc1 = -1;
    yinc2 = -1;
  }
  if (deltax >= deltay) //There is at least one x-value for every y-value
  {
    xinc1 = 0; //Don't change the x when numerator >= denominator
    yinc2 = 0; //Don't change the y for every iteration
    den = deltax;
    num = deltax / 2;
    numadd = deltay;
    numpixels = deltax; //There are more x-values than y-values
  }
  else //There is at least one y-value for every x-value
  {
    xinc2 = 0; //Don't change the x for every iteration
    yinc1 = 0; //Don't change the y when numerator >= denominator
    den = deltay;
    num = deltay / 2;
    numadd = deltax;
    numpixels = deltay; //There are more y-values than x-values
  }
  for (curpixel = 0; curpixel <= numpixels; curpixel++)
  {
    Gpset(x, y);  //Draw the current pixel
    num += numadd; //Increase the numerator by the top of the fraction
    if (num >= den) //Check if numerator >= denominator
    {
      num -= den; //Calculate the new numerator value
      x += xinc1; //Change the x as appropriate
      y += yinc1; //Change the y as appropriate
    }
    x += xinc2; //Change the x as appropriate
    y += yinc2; //Change the y as appropriate
  }
  return 1;
}


bool Gcircle(int xc, int yc, int radius)
{
  if(xc - radius < 0 || xc + radius >= w || yc - radius < 0 || yc + radius >= h) return 0;
  int x = 0;
  int y = radius;
  int p = 3 - (radius << 1);
  int a, b, c, d, e, f, g, h;
  while (x <= y)
  {
     a = xc + x; //8 pixels can be calculated at once thanks to the symmetry
     b = yc + y;
     c = xc - x;
     d = yc - y;
     e = xc + y;
     f = yc + x;
     g = xc - y;
     h = yc - x;
     Gpset(a, b);
     Gpset(c, d);
     Gpset(e, f);
     Gpset(g, f);
     if(x>0) //avoid drawing pixels at same position for correct transparency
     {
       Gpset(a, d);
       Gpset(c, b);
       Gpset(e, h);
       Gpset(g, h);
     }
     if(p < 0) p += (x++ << 2) + 6;
     else p += ((x++ - y--) << 2) + 10;
  }
  
  return 1;
}

bool GhorLine(int y, int x1, int x2)
{
  if(x2 < x1) {x1 += x2; x2 = x1 - x2; x1 -= x2;} //swap x1 and x2, x1 must be the leftmost endpoint   
  if(x2 < 0 || x1 > w - 1 || y < 0 || y > h - 1) return false; //no single point of the line is on screen
  
  if(x1 < 0) x1 = 0;
  if(x2 >= w) x2 = w - 1;
  
  for(int x = x1; x < x2; x++)
  {
    Gpset(x, y);
  }
  
  return true;
}

//Filled bresenham circle with center at (xc,yc) with radius and red green blue color
bool Gdisk(int xc, int yc, int radius)
{
  if(xc + radius < 0 || xc - radius >= w || yc + radius < 0 || yc - radius >= h) return 0; //every single pixel outside screen, so don't waste time on it
  int x = 0;
  int y = radius;
  int p = 3 - (radius << 1);
  int a, b, c, d, e, f, g, h;
  int pb = 0, pd = 0; //previous values: to avoid drawing horizontal lines multiple times, screwing up the transparency
  while (x <= y)
  {
     // write data
     a = xc + x;
     b = yc + y;
     c = xc - x;
     d = yc - y;
     e = xc + y;
     f = yc + x;
     g = xc - y;
     h = yc - x;
     if(b != pb) GhorLine(b, a, c);
     if(d != pd) GhorLine(d, a, c);
     if(f != b) GhorLine(f, e, g);
     if(h != d && h != f) GhorLine(h, e, g);
     pb = b;
     pd = d;
     if(p < 0) p += (x++ << 2) + 6;
     else p += ((x++ - y--) << 2) + 10;
  }
  
  return 1;
}

void Gcls()
{
  for(int x = 0; x < w; x++)
  for(int y = 0; y < w; y++)
  {
    screenBuffer[h * x + y] = 0;
  }  
}  

bool Grect(int x1, int y1, int x2, int y2)
{
  if(x1 < 0 || x1 > w - 1 || x2 < 0 || x2 > w - 1 || y1 < 0 || y1 > h - 1 || y2 < 0 || y2 > h - 1) return 0;
  
  for(int x = x1; x < x2; x++)
  for(int y = y1; y < y2; y++)
  {
    Gpset(x, y);
  }
  
  return 1;
}  

static void inline GdrawLetter(unsigned char n, int x, int y)
{
  int u, v;

  for (v = 0; v < 8; v++)
  for (u = 0; u < 8; u++)
  {
    if(font[n][u][v]) Gpset(x + u, y + v);
    else if(int(sc[12]) & 4096) 
    Gpset(x + u, y + v, 1);
  }
}

//Draws a string of text
int Gprint(const char* text, int x, int y)
{
  int pos=0;
  while(text[pos])
  {
     GdrawLetter(text[pos++], x, y);
     x += 8;
     if(x > w - 8) {x %= 8; y += 8;}
     if(y > h - 8) {y %= 8;}
  }
  return pos;
}

//Draws an integer number
int Giprint(int a, int x, int y, bool orderSigns = 0)
{
  char text[255];
  std::sprintf(text, "%i", a);
  int negationSymbol = 0;
  if(a >= 0 && orderSigns) negationSymbol = 8;
  return Gprint(text, x + negationSymbol, y);
}

//Draws a floating point number
int Gfprint(double a, int length, int x, int y, bool orderSigns = 0)
{
  char text[255];
  switch(length)
  {
    case 1: std::sprintf(text, "%.1f", a); break;
    case 2: std::sprintf(text, "%.2f", a); break;
    case 3: std::sprintf(text, "%.3f", a); break;
    case 4: std::sprintf(text, "%.4f", a); break;
    case 5: std::sprintf(text, "%.5f", a); break;
    case 6: std::sprintf(text, "%.6f", a); break;
    case 7: std::sprintf(text, "%.7f", a); break;
    case 8: std::sprintf(text, "%.8f", a); break;
    case 9: std::sprintf(text, "%.9f", a); break;
    case 10: std::sprintf(text, "%.10f", a); break;
    case 11: std::sprintf(text, "%.11f", a); break;
    case 12: std::sprintf(text, "%.12f", a); break;
    case 13: std::sprintf(text, "%.13f", a); break;
    case 14: std::sprintf(text, "%.14f", a); break;
    case 15: std::sprintf(text, "%.15f", a); break;
    default: std::sprintf(text, "%.16g", a); break; 
  }
  int negationSymbol = 0;
  if(a >= 0 && orderSigns) negationSymbol = 8;
  return Gprint(text, x + negationSymbol, y);
}

//Draws a single character (actually the same as drawletter)
int Gcprint(unsigned char n, int x, int y)
{
   GdrawLetter(n, x, y);
   return 1;
}

////////////////////////////////////////////////////////////////////////////////

void initializeParameters()
{
  //initialize buffers and values
  for(int x = 0; x < MAXIW; x++)
  for(int y = 0; y < MAXIH; y++)
  {
    instruction[x][y] = 0;
  }
  for(int x = 0; x < MAXGOSUB; x++)
  {
    gosubx[x] = 0;
    gosuby[x] = 0;
    gosubd[x] = 0;
  } 
  for(int x = 0; x < RA; x++)
  {
    sc[x] = 0.0;
  } 
  sc[2] = sc[3] = sc[4] = 255;
  for(int x = 0; x < SA; x++)
  {
    st[x] = 0.0;
  } 
  for(int x = 0; x < MAXSCREENU; x++)
  for(int y = 0; y < MAXSCREENV; y++)
  {
    GsetRGB(screenBuffer[MAXSCREENV * x + y], 0, 0, 0);
  }
  ix = 0;
  iy = 0;
  id = 1; //instruction pointer direction (N, E, S, W); 1 = E (to the right)
  rp = 0; 
  sp = 0;
  dc = 1; //the decimal counter: 1 = will create new number on stack, 0 = enter digits before point, 0.1 - 0.0000000...00001 = enter digits after the point
  anchorTime = 5; //minimum time between two time anchors in milliseconds
  lastAnchorTime = getTicks(); //time at previous time anchor
  instructionMode = 0; //extended or string mode: 0 = normal, 1 = extended command, 2 = string mode, 3 = temporary normal command mode while in string mode
  debugMode = 0;
  gp = 0; //gosub pointer
  ticks = 0;
  numStackProgram = 0; //hoeveel registers / stack kopieren met stack program
  for(int i = 0; i < 16; i++) stackProgram[i] = 0; //16 register addressen voor te poppen / pushen
  screen(256,256,0, "Gammaplex - Hit esc for menu");
}

int checkUserInterpreterInput(bool delay)
{
  int result = 0;
  if(done(false, delay)) result = 1;
  //when you press escape, give options
  readKeys();
  if(keyPressed(SDLK_ESCAPE)) result = 2;
  return result;
}

void userInteraction();
bool finish = false;

void executeCommand(Uint8 executeInstruction)
{
  //set decimal counter to 1 if command is not one of the following:
  if(((currentInstruction < '0' || currentInstruction > '9') &&
      currentInstruction != '.' && currentInstruction != '\\'&&
      currentInstruction != '/' && currentInstruction != '@' &&
      currentInstruction != ';' && currentInstruction != ' ' &&
      currentInstruction != '<' && currentInstruction != '>' &&
      currentInstruction != '^' && currentInstruction != 'v' &&
      currentInstruction != 0)) dc = 1;
  
  
  //perform current command
  bool executeStringCommand = 0;
  if(instructionMode == 3 &&
     (currentInstruction == '>' || currentInstruction == 'v' ||
      currentInstruction == '<' || currentInstruction == '^' ||
      currentInstruction == '/' || currentInstruction == '\\'||
      currentInstruction == ';' || currentInstruction == ' ' ||
      currentInstruction == '#' || currentInstruction == '@' ||
      currentInstruction == ';' || currentInstruction == ' ' ||
      currentInstruction == '+' || currentInstruction == '-' ||
      currentInstruction == '*' || currentInstruction == ':'))
      executeStringCommand = 1;
  
  if(instructionMode == 0 || executeStringCommand)
  {
    switch(executeInstruction)
    {
      //setting and getting rp and sp
      case 'u': rp++; break;
      case 'd': rp--; break;

      //program flow
      case 92: if(id == 0) id = 3;
           else if(id == 3) id = 0;
           else if(id == 1) id = 2;
           else id = 1;
           break;
      case '/': if(id == 0) id = 1;
           else if(id == 1) id = 0;
           else if(id == 2) id = 3;
           else id = 2;
           break;
      case '^': id = 0; break;
      case '>': id = 1; break;
      case 'v': id = 2; break;
      case '<': id = 3; break;
      case 'G': ix = int(pop()); iy = int(pop()); 
            moveIPBack();
            break;
      case 'g': push(iy); push(ix); break;
      case '?':
       {
         double u = pop();
         if(u == 0) moveIP(); //if current stack is false, jump one more forward
         break;
       }
      case ';': moveIP(); break; //jump over next command and do not execute it, the second moveIP() is done by general program flow ofcourse
      case 'E':
        {
         int user = checkUserInterpreterInput(true);
         if(user == 1) finish = true;
         else if(user == 2) userInteraction();
         moveIPBack();
         break;
        }
      case 'e': sleep(); break;
      case 'j': while(getTicks() - lastAnchorTime < anchorTime)
            {
              SDL_Delay(1);
              if(done(false)) end();
            }
            lastAnchorTime = getTicks();
            break;

      //math
      case '0': changeDC(0); break;
      case '1': changeDC(1); break;
      case '2': changeDC(2); break;
      case '3': changeDC(3); break;
      case '4': changeDC(4); break;
      case '5': changeDC(5); break;
      case '6': changeDC(6); break;
      case '7': changeDC(7); break;
      case '8': changeDC(8); break;
      case '9': changeDC(9); break;
      case '.': if(dc == 1) push(0); dc = 0.1; break;
      case '"': st[sp]++; break;
      case '\'': st[sp]--; break;
      case '+': { double s = pop(); double t = pop(); push(s + t); break; }
      case '-': { double s = pop(); double t = pop(); push(t - s); break; }
      case '*': { double s = pop(); double t = pop(); push(s * t); break; }
      case ':': { double s = pop(); double t = pop(); push(t / s); break; }
      case '~': { double s = pop(); double t = pop(); push(pow(t, s)); break; }
      case 'V': st[sp] = sqrt(st[sp]); break;
      case 'T': push(std::cos(pop())); break;
      case 'p': push(3.1415926535897932384626433832795); break;
      case '%': { double s = pop(); double t = pop(); push(double(int(s) % std::max(1, int(t)))); break; }
      case '&': { double s = pop(); double t = pop(); push(double(int(s) & int(t))); break; }
      case '|': { double s = pop(); double t = pop(); push(double(int(s) | int(t))); break; }
      case 'x': { double s = pop(); double t = pop(); push(double(int(s) ^ int(t))); break; }
      case '!': { double s = pop(); if(s == 0) s = 1; else s = 0; push(s); break; }
      case '_': { st[sp] = -st[sp]; break; }
      case '=': { double s = pop(); double t = pop(); if(s == t) push(1); else push(0); break; }
      case ',': { double s = pop(); double t = pop(); if(s < t) push(1); else push(0); break; }
      case 't': push(double(getTicks())); break;
      case 'N': st[sp] = 0; break;
      case 'n': push(255); break;
      case 'o': push(int(pop())); break;
      case 'k': push(abs(std::rand()) % 2); break;
      case 'K': push((abs(std::rand()) % 16384) / 16384.0); break;

      //input
      case 'M': { int x, y; getMouseState(x, y); push(y); push(x); break; }
      case 'm': { int x, y; bool b1, b2; getMouseState(x, y, b1, b2); push(b2); push(b1); break; }
      case 'I':
      {
        std::string message;
        int m = Uint32(pop());
        if(m == 1) message = ">"; else if(m == 2) message = "]"; else if(m == 3) message = "enter char: "; else if(m == 4) message = "please enter a character: ";
        else if(m == 5) message = "enter key: "; else if(m == 6) message = "please enter a key: ";
        std::string input = getInput<std::string>(message, false, Uint32(sc[0]) % w, Uint32(sc[1]) % h, ColorRGB(Uint8(sc[2]), Uint8(sc[3]), Uint8(sc[4])));
        if(!input.empty()) push(input[0]);
        else push(0);
        break;
      }
      case 'J':
      {
        std::string message;
        int m = Uint32(pop());
        if(m == 1) message = ">"; else if(m == 2) message = "]"; else if(m == 3) message = "enter number: "; else if(m == 4) message = "please enter a number: ";
        else if(m == 5) message = "enter value: "; else if(m == 6) message = "please enter a value: ";
        push(getInput<double>(message, false, Uint32(sc[0]) % w, Uint32(sc[1]) % h, ColorRGB(Uint8(sc[2]), Uint8(sc[3]), Uint8(sc[4]))));
        break;
      }

      //debug mode
      case 'U': debugMode++; debugMode %= 4; break;

      //self modifying code
      case 'q':
       {
         int x = Uint32(pop()) % iw; int y = Uint32(pop()) % ih;
         push(instruction[x][y]);
         break;
       }
      case 'Q':
        {
          int x = Uint32(pop()) % iw; int y = Uint32(pop()) % ih;
          instruction[x][y] = Uint8(pop());
          break;
        }

      //graphics
      case 'l': screen(std::min(std::max(Uint32(sc[0]), (Uint32)64), (Uint32)MAXSCREENU), std::min(std::max((Uint32)(sc[1]), (Uint32)64), (Uint32)MAXSCREENV), 0 ,"Gammaplex - Hit esc for menu"); break;
      case 'P': Gpset(Uint32(sc[0]) % w, Uint32(sc[1]) % h); break;
      case 'L':
       {
         int x1 = int(sc[0]); int y1 = int(sc[1]); int x2 = int(sc[6]); int y2 = int(sc[7]);
         int x3, y3, x4, y4;
         if(clipLine(x1, y1, x2, y2, x3, y3, x4, y4))
           Gline(x3, y3, x4, y4);
         break;
       }
      case 'R': drawBuffer(screenBuffer);
            if(debugMode > 0) printDebugInfo(); //debug info is printed to screenbuffer, not gamma graphics buffer
            redraw();
            break;
      case 'h': push(h-1); break;
      case 'y': push(w-1); break;
      case 'c': Gcls(); break;
      case 'C': Gcircle(Uint32(sc[0]) % w, Uint32(sc[1]) % h, Uint32(sc[5]));
            break;
      case 'F': Gdisk(Uint32(sc[0]) % w, Uint32(sc[1]) % h, Uint32(sc[5]));
            break;
      case 'B':
       {
         int x1 = int(sc[0]); int y1 = int(sc[1]); int x2 = int(sc[6]); int y2 = int(sc[7]);
         int x3, y3, x4, y4;
         if(clipLine(x1, y1, x2, y2, x3, y3, x4, y4)) //clipline is just as useful for clipping rectangles
           Grect(x3, y3, x4, y4);
         break;
       }
      case 'r': Gcprint(Uint8(pop()), Uint32(sc[0]) % w, Uint32(sc[1]) % h);
            sc[0] += 8;
            if(sc[0] > w - 8) {sc[0] = Uint32(sc[0]) % 8; sc[1] += 8;}
            if(sc[1] > h - 8) {sc[1] = Uint32(sc[1]) % 8;}
            break;
      case 'i': push(Giprint(Sint32(pop()), Uint32(sc[0]) % w, Uint32(sc[1]) % h));
            break;
      case 'f': push(Gfprint(pop(), 6, Uint32(sc[0]) % w, Uint32(sc[1]) % h));
            break;
      case 'H':
        {
          ColorHSV hsv;
          hsv.h = int(pop()); if(hsv.h < 0) hsv.h = 0; if(hsv.h > 255) hsv.h = 255;
          hsv.s = int(pop()); if(hsv.s < 0) hsv.s = 0; if(hsv.s > 255) hsv.s = 255;
          hsv.v = int(pop()); if(hsv.v < 0) hsv.v = 0; if(hsv.v > 255) hsv.v = 255;
          ColorRGB rgb = HSVtoRGB(hsv);
          push(rgb.b);
          push(rgb.g);
          push(rgb.r);
        }
        break;
      //stack
      case '(': push(sc[rp]); break;
      case ')': sc[rp] = pop(); break;
      case '[': push(rp); break;
      case ']': rp = Uint32(pop()); break;
      case 'D': pop(); break;
      case 'w': push(st[sp]); break;
      case 'W': push(st[(sp-1)%SA]); push(st[(sp-1)%SA]); break;
      case 'Y':
        {
          int i = int(pop()); 
          for(int ii = 0; ii < i; ii++) push( st[ (sp-i+1)%SA ] );
          break;
        }
      case 's': st[(sp-1)%SA] += st[sp];  st[sp] = st[(sp-1)%SA] - st[sp];  st[(sp-1)%SA] -= st[sp]; break;
      case 'S': st[(sp-2)%SA] += st[sp];  st[sp] = st[(sp-2)%SA] - st[sp];  st[(sp-2)%SA] -= st[sp]; break;
      case '$': { int i = int(pop()); st[(sp-i)%SA] += st[sp];  st[sp] = st[(sp-i)%SA] - st[sp];  st[(sp-i)%SA] -= st[sp]; break; }
      case 'z':
      {
        double u = st[sp];
        st[sp] = st[(sp - 2) % SA];
        st[(sp - 2) % SA] = st[(sp - 1) % SA];
        st[(sp - 1) % SA] = u;
        break;
      }
      case 'Z':
       {
         int j = int(pop());
         double u = st[sp];
         st[sp] = st[(sp - j) % SA];
         for(int jj = j; jj >= 2; jj++) st[(sp - jj) % SA] = st[(sp - jj + 1) % SA];
         st[(sp - 1) % SA] = u;
         break;
       }
      case '{': for(int g = 0; g < numStackProgram; g++)
            {
              push(sc[(numStackProgram - stackProgram[g] - 1) % RA]);
            }
            break;
      case '}': for(int g = 0; g < numStackProgram; g++)
            {
              sc[stackProgram[g] % RA] = pop();
            }
            break;
      case 'a':
        {
          int x = abs(int(pop()));
          switch(x)
          {
            case 0: numStackProgram = 0;
                break;
            case 1: numStackProgram = 2;
                stackProgram[0] = 0;
                stackProgram[1] = 1;
                break;
            case 2: numStackProgram = 2;
                stackProgram[0] = 6;
                stackProgram[1] = 7;
                break;
            case 3: numStackProgram = 3;
                stackProgram[0] = 2;
                stackProgram[1] = 3;
                stackProgram[2] = 4;
                break;
            case 4: numStackProgram = 3;
                stackProgram[0] = 8;
                stackProgram[1] = 9;
                stackProgram[2] = 10;
                break;
            case 5: numStackProgram = 8;
                stackProgram[0] = 0;
                stackProgram[1] = 1;
                stackProgram[2] = 2;
                stackProgram[3] = 3;
                stackProgram[4] = 4;
                stackProgram[5] = 5;
                stackProgram[6] = 6;
                stackProgram[7] = 7;
                break;
            case 6: numStackProgram = 13;
                stackProgram[0] = 0;
                stackProgram[1] = 1;
                stackProgram[2] = 2;
                stackProgram[3] = 3;
                stackProgram[4] = 4;
                stackProgram[5] = 5;
                stackProgram[6] = 6;
                stackProgram[7] = 7;
                stackProgram[8] = 8;
                stackProgram[9] = 9;
                stackProgram[10] = 10;
                stackProgram[11] = 11;
                stackProgram[12] = 12;
                break;
          }
          break;
        }
      case 'A':
        {
          int x = abs(int(pop()));
          int y = 0;
          if(x > 16) x = 16;
          while(y < x)
          {
            y++;
            stackProgram[y] = std::max(0, std::min(RA, int(pop())));
          }
          break;
        }
      case 'X': instructionMode = 1; break;
      default: break;
    }
    if(instructionMode == 3) 
    {
      if(stringCommandLifeSpan == 0) instructionMode = 2;
      if(stringCommandLifeSpan > 0) stringCommandLifeSpan--;
    }
  }
  else if(instructionMode == 1) //extended instructions
  {
    instructionMode = 0; //the next instruction has to be unextended again
    switch(executeInstruction)
    {
      case ';': moveIP(); moveIP(); break; //jump over next command aand the one after that nd do not execute it, the third moveIP() is done by general program flow ofcourse
      case 92: if(id == 0) id = 3;
           else if(id == 3) id = 0;
           else if(id == 1) id = 2;
           else id = 1;
           instructionMode = 1;
           break;
      case '/': if(id == 0) id = 1;
           else if(id == 1) id = 0;
           else if(id == 2) id = 3;
           else id = 2;
           instructionMode = 1;
           break;
      case '^': id = 0; instructionMode = 1; break;
      case '>': id = 1; instructionMode = 1; break;
      case 'v': id = 2; instructionMode = 1; break;
      case '<': id = 3; instructionMode = 1; break;
      case ' ': instructionMode = 1;  break;
      case '@': instructionMode = 1; break;
      case '#': instructionMode = 0; break; 
      case '"': instructionMode = 2; break;
      case 'U': debugMode = 0; break;
      //GOSUB
      case 'G': gosubx[gp] = ix;
            gosuby[gp] = iy;
            gosubd[gp] = id;
            gp++;
            gp %= MAXGOSUB;
            ix = int(pop()); iy = int(pop()); 
            moveIPBack();
            break;
      case 'g': gp--;
            if(gp < 0) gp = MAXGOSUB - 1;
            ix = gosubx[gp]; iy = gosuby[gp]; id = gosubd[gp];
            //moveIPBack(); //NO WAY, don't moveIPBack for "return" after gosub, or you'll do the gosub command AGAIN! ;)
            break;
      case 'j': anchorTime = std::max(0, int(pop())); break;
      case '?':
        {
          double u = pop();
          if(u == 0) {moveIP(); moveIP();}//if current stack is false, jump one more forward
          break;
        }
      case 'H':
        {
          ColorRGB rgb;
          rgb.r = int(pop()); if(rgb.r < 0) rgb.r = 0; if(rgb.r > 255) rgb.r = 255;
          rgb.g = int(pop()); if(rgb.g < 0) rgb.g = 0; if(rgb.g > 255) rgb.g = 255;
          rgb.b = int(pop()); if(rgb.b < 0) rgb.b = 0; if(rgb.b > 255) rgb.b = 255;
          ColorHSV hsv = RGBtoHSV(rgb);
          push(hsv.h);
          push(hsv.s);
          push(hsv.v);
        }
        break;
      case 'T':
        {
          int r = abs(int(pop())); //Trigonometri
          double u = pop();
          switch(r)
          {
            case 0: push(0); break;
            case 1: push(sin(u)); break;
            case 2: push(cos(u)); break;
            case 3: push(tan(u)); break;
            case 4: push(1.0 / cos(u)); break;
            case 5: push(1.0 / sin(u)); break;
            case 6: push(1.0 / tan(u)); break;
            case 7: push(asin(u)); break;
            case 8: push(acos(u)); break;
            case 9: push(atan(u)); break;
            case 10: push(1.0 / acos(u)); break;
            case 11: push(1.0 / asin(u)); break;
            case 12: push(1.0 / atan(u)); break;
            case 13: push(sinh(u)); break;
            case 14: push(cosh(u)); break;
            case 15: push(tanh(u)); break;
            case 16: push(1.0 / cosh(u)); break;
            case 17: push(1.0 / sinh(u)); break;
            case 18: push(1.0 / tanh(u)); break;
            case 19: push(log(u + sqrt(1 + u * u))); break;
            case 20: push(log(u + sqrt(u + 1) * sqrt(u - 1))); break;
            case 21: push(0.5 * (log(1 + u) - log(1 - u))); break;
            case 22: push(log(sqrt(1 / u - 1) * sqrt(1 / u + 1) + 1 / u)); break;
            case 23: push(log(sqrt(1 + 1 / (u * u)) + 1 / u)); break;
            case 24: push(0.5 * (log(1 + 1 / u) - log(1 - 1 / u))); break;
            case 25: push(exp(u)); break;
            case 26: push(log(u)); break;
            case 27: push(exp(-(u * u))); break; //sorta gaussian distrubution (standard normal distribution)
            case 28: if(u < 0) push(-1);
                  else if(u == 0) push(0);
                  else push(1);
                  break;
            case 64: push(atan2(pop(), u)); break;
            case 65: push(log(pop())/log(u)); break;
            default: break;
          }
          break;
        }
      case 's':
        {
          int x = abs(int(pop())) % SA;
          for(int gg = 0; gg < x / 2; gg++)
          {
            st[(sp - gg) % SA] += st[(sp - x + gg + 1) % SA];  st[(sp - x + gg + 1) % SA] = st[(sp - gg) % SA] - st[(sp - x + gg + 1) % SA];  st[(sp - gg) % SA] -= st[(sp - x + gg + 1) % SA];
          }
          break;
        }
      case 'S':
        {
          int y = sp;
          int x = 0;
          while(st[y] > 0 && st[y] < 256 && x < SA) 
          {
            y--;
            if(y < 0) y = SA - 1;
            x++;
          }
          for(int gg = 0; gg < x / 2; gg++)
          {
            st[(sp - gg) % SA] += st[(sp - x + gg + 1) % SA];  st[(sp - x + gg + 1) % SA] = st[(sp - gg) % SA] - st[(sp - x + gg + 1) % SA];  st[(sp - gg) % SA] -= st[(sp - x + gg + 1) % SA];
          }
          break;
        }
      case 'w':
        {
          std::string text;
          text.resize(256);
          int y = sp;
          int x = 0;
          while(st[y] > 0 && st[y] < 256 && x < SA) 
          {
            y--;
            if(y < 0) y = SA - 1;
            x++;
          }
          for(int gg = 0; gg < x; gg++)
          {
            text[gg] = Uint8(st[(sp - gg) % SA]);
          }
          push(0);
          for(int gg = x - 1; gg >= 0; gg--)
          {
            push(text[gg]);
          }
          break;
        }
      case 'I':
        {
          std::string message;
          int m = Uint32(pop());
          if(m == 1) message = ">"; else if(m == 2) message = "]"; else if(m == 3) message = "enter text: "; else if(m == 4) message = "please enter a text: ";
          else if(m == 5) message = "enter string: "; else if(m == 6) message = "please enter a string: ";
          std::string text = getInput<std::string>(message, false, Uint32(sc[0]) % w, Uint32(sc[1]) % h, ColorRGB(Uint8(sc[2]), Uint8(sc[3]), Uint8(sc[4])));
          push(0);
          for(int i = text.size() - 1; i >= 0; i--) push(text[i]);
        }
        break;
      case 'r':
        {
          int x = 0;
          while(st[sp] > 0 && st[sp] < 256 && x < SA)
          {
            Gcprint(Uint8(pop()), Uint32(sc[0]) % w, Uint32(sc[1]) % h);
            sc[0] += 8;
            if(sc[0] > w - 8) {sc[0] = Uint32(sc[0]) % 8; sc[1] += 8;}
            if(sc[1] > h - 8) {sc[1] = Uint32(sc[1]) % 8;}
            x++;
          }
          //pop(); 
          break;
        }
      case 'P':
        {
          Uint8 red, green, blue;
          GgetRGB(screenBuffer[h * (abs(int(sc[0])) % w) + (abs(int(sc[1])) % h)], red, green, blue);
          push(red); push(green); push(blue);
          break;
        }
      default: break;
    }
  }
  else if(instructionMode == 2)
  {
    if(executeInstruction == '"') instructionMode = 3;
    else push(executeInstruction);
  }
  else if(instructionMode == 3)
  {
    if(stringCommandLifeSpan == 0) instructionMode = 2;
    if(stringCommandLifeSpan > 0) stringCommandLifeSpan--;
    if(executeInstruction == 'X') instructionMode = 0;
    if(executeInstruction == '"') push('"');
    if(executeInstruction == '2') {stringCommandLifeSpan = 1; instructionMode = 3;}
  }
}

void userInteraction()
{
  cls();
  int i = -1;
  bool chosen = 0;
  
  while(!chosen)
  {
    print("Gammaplex Programming Language", 0, 0, ColorRGB(255, 0, 0), 1);
    print("Choose Option:", 0, 8,ColorRGB(255,255,255),1);
    print(" 1. Continue", 0, 16,ColorRGB(255,255,255),1);
    print(" 2. Exit", 0, 24,ColorRGB(255,255,255),1);
    print(" 3. Load Code File", 0, 32,ColorRGB(255,255,255),1);
    print(" 4. Memory Editor", 0, 40,ColorRGB(255,255,255),1);
    print(" 5. Debugging Options", 0, 48,ColorRGB(255,255,255),1);
    print(" 6. Execute Command", 0, 56,ColorRGB(255,255,255),1); 
    print(" 9. About...", 0, 64,ColorRGB(255,255,255),1);
    redraw();
    i = getInputCharacter() - '0';
    if(i > 0 && i < 10) chosen = 1;
    readKeys();
    if(keyPressed(SDLK_ESCAPE)) chosen = 1;
    if(done(false)) end();
  }
  if(i == 2)
  {
    end();
  }
  if(i == 3)
  {
    initializeParameters();
    std::string filename;
    getInputString(filename, "Enter filemane: ");
    loadCodeFile(filename.c_str());
    cls();
  }
  if(i == 4)
  {
    cls();
    chosen = 0;
    bool editloop = 1;
    int j = -1;
    while(editloop)
    {
      print("Choose Option",0,0,ColorRGB(255,255,255),1);
      print(" 0. Continue", 0, 8,ColorRGB(255,255,255),1);
      print(" 1. Show Info", 0, 16,ColorRGB(255,255,255),1);
      print(" 2. Modify Register", 0, 24,ColorRGB(255,255,255),1);
      print(" 3. Pop Stack", 0, 32,ColorRGB(255,255,255),1);
      print(" 4. Push Stack", 0, 40,ColorRGB(255,255,255),1);
      print(" 5. Modify Instruction", 0, 48,ColorRGB(255,255,255),1);
      print(" 6. Modify Instruction Array", 0, 56,ColorRGB(255,255,255),1);
      print(" 7. Modify IP", 0, 64,ColorRGB(255,255,255),1);
      redraw();
      j = getInputCharacter() - '0';
      if(j == 0) editloop = 0;
      if(j == 1)
      {
        cls();
        print("iw:", 0, 0); print(iw, 24, 0,ColorRGB(255,255,255),1);
        print("ih:", 0, 8); print(ih, 24, 8,ColorRGB(255,255,255),1);
        print("ix:", 0, 16); print(ix, 24, 16,ColorRGB(255,255,255),1);
        print("iy:", 0, 24); print(iy, 24, 24,ColorRGB(255,255,255),1);
        print("id:", 0, 32); print(id, 24, 32,ColorRGB(255,255,255),1);
        print("Press any key to continue", 0, 40,ColorRGB(255,255,255),1);
        redraw();
        sleep();
        cls();
      }
      if(j == 2)
      {
        Uint32 reg = getInput<Uint32>("Enter Register Number: ", false, 0, 0, RGB_White, true) % RA;
        sc[reg] = getInput<Uint32>("Enter New Value: ", false, 0, 0, RGB_White, true);
        cls();
      }
      if(j == 3) pop();
      if(j == 4) 
      {
        push(getInput<double>("Enter Value to Push: ", false, 0, 0, RGB_White, true));
        cls();
      }
      if(j == 5)
      {
        int inx = Uint32(getInput<Uint32>("Enter X Coordinate: ", false, 0, 0, RGB_White, true))%iw;
        int iny = Uint32(getInput<Uint32>("Enter Y Coordinate: ", false, 0, 0, RGB_White, true))%ih;
        instruction[inx][iny] = Uint8(getInput<Uint8>("Enter X Coordinate: ", false, 0, 0, RGB_White, true));
        cls();
      } 
      if(j == 6)
      {
        iw = std::max((Uint32)1, Uint32(getInput<Uint32>("Enter X Size: ", false, 0, 0, RGB_White, true)) % MAXIW);
        ih = std::max((Uint32)1, Uint32(getInput<Uint32>("Enter Y Size: ", false, 0, 0, RGB_White, true)) % MAXIH);
        cls();
      }
      if(j == 7)
      {
        ix = Uint32(getInput<Uint32>("Enter X Coordinate: ", false, 0, 0, RGB_White, true)) % iw;
        iy = Uint32(getInput<Uint32>("Enter Y Coordinate: ", false, 0, 0, RGB_White, true)) % ih;
        id = Uint8(getInput<Uint8>("Enter Direction: ", false, 0, 0, RGB_White, true)) % 4;
        cls();
      }
      if(keyPressed(SDLK_ESCAPE)) chosen = 1;
      if(done(false)) end();
    }
  }
  if(i == 5)
  {
    cls();
    chosen = 0;
    int j;
    while(!chosen)
    {
      print("Choose Debug Mode",0,0,ColorRGB(255,255,255),1);
      print(" 0. No Debug Mode", 0, 8,ColorRGB(255,255,255),1);
      print(" 1. Show Info At Redraw", 0, 16,ColorRGB(255,255,255),1);
      print(" 2. Show Info Every Tick", 0, 24,ColorRGB(255,255,255),1);
      print(" 3. Sleep Every Tick", 0, 32,ColorRGB(255,255,255),1);
      print(" 4. Continue", 0, 40,ColorRGB(255,255,255),1);
      redraw();
      j = getInputCharacter() - '0';
      if(j > -1 && j < 4) {debugMode = j; chosen = 1;}
      if(j == 4) chosen = 1;
      readKeys();
      if(keyPressed(SDLK_ESCAPE)) chosen = 1;
      if(done(false)) end();
    }
  }
  if(i == 6)
  {
    bool finished = 0;
    cls();
    chosen = 0;
    int commandEnterMode = 0;
    int j;
    while(!chosen)
    {
      print("How to enter commands?",0,0,ColorRGB(255,255,255),1);
      print(" 0. Continue", 0, 8,ColorRGB(255,255,255),1);
      print(" 1. Type Character", 0, 16,ColorRGB(255,255,255),1);
      print(" 2. Type ASCII Value", 0, 24,ColorRGB(255,255,255),1);
      redraw();
      j = getInputCharacter() - '0';
      if(j == 0) {chosen = 1; commandEnterMode = 0;}
      if(j == 1) {chosen = 1; commandEnterMode = 1;}
      if(j == 2) {chosen = 1; commandEnterMode = 2;}
      readKeys();
      if(keyPressed(SDLK_ESCAPE)) chosen = 1;
      if(done(false)) end();
    }
    while(!finished && commandEnterMode > 0)
    {
      if(done(false)) end();
      readKeys();
      if(keyPressed(SDLK_ESCAPE)) finished = 1;
      Uint8 command = 0;

      if(commandEnterMode == 1)
      {
        std::string input = getInput<std::string>("Enter command, E ends: ", true, 0, 0, RGB_Yellow, true);
        command = input.empty() ? 0 : input[0];
      }
      if(commandEnterMode == 2)
      {
        command = getInput<int>("Enter command number, 0 ends: ", true, 0, 0, RGB_Yellow, true);
      }
      if(command == 0) finished = 1;
      if(commandEnterMode == 1 && command == 'E') finished = 1;
      else if(command >= 32 && command < 128) executeCommand(command);
    }
    drawBuffer(screenBuffer);
    redraw();
  }
  if(i == 9)
  {
    cls();
    print("Gammaplex Programming Language v0.34 Interpreter v20071031. (c) 2004-2007 Lode Vandevenne. See included manual or go to http://members.gamedev.net/lode/projects/esolangs/gammaplex/index.html for more information.",0,0,ColorRGB(255,255,255),1);
    redraw();
    sleep();
    cls();
  }


  drawBuffer(screenBuffer);
  redraw();
}

int main(int argc, char *argv[])
{
  (void)argc;
  (void)argv;
  
  screen(256,256,0, "Gammaplex - Hit esc for menu"); 
  
  initializeParameters();
  
  if(argv[1]) loadCodeFile(argv[1]);
  else loadCodeFile("gamma.txt");
  
  //set initial anchor time
  lastAnchorTime = getTicks();
  
  cls();
  redraw();

  //start main loop
  int numcommandcounter = 0;
  while(!finish) //finish is global because executecommand can also change it
  {
    numcommandcounter++;
    if(numcommandcounter > 4096) //do the SDL event stuff only after every so many commands...
    {
      numcommandcounter = 0;
      int user = checkUserInterpreterInput(false);
      if(user == 1) finish = true;
      else if(user == 2) userInteraction();
    }

    //get current instruciton
    currentInstruction = instruction[ix][iy];

    //safety for array parameters
    if(/*sp < 0 ||*/ sp >= (Uint32)SA) sp %= SA;
    if(/*rp < 0 ||*/ rp >= (Uint32)RA) rp %= RA;

    executeCommand(currentInstruction);

    //increment instruction pointer in current direction
    moveIP();
    ticks++;

    //debug info
    if(debugMode > 0)
    {
      //when debugmode is 1: printed at redraw command 'R'
      if(debugMode >= 2) //do it every single frame now
      {
        drawBuffer(screenBuffer);
        printDebugInfo();
        redraw();
      }
      if(debugMode == 3) sleep();
    }
  }
  
  return(0);
}
