//Deltaplex Interpreter v 1.0

/*
Copyright (c) 2005-2006 Lode Vandevenne
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
  * Neither the name of Lode Vandevenne nor the names of his contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*
g++ *.cpp -lSDL -lGL -ansi -pedantic -W -Wall
*/


/*
This is the main.cpp file, which contains the interpreter of deltaplex.

All other .cpp and .h files are just general library wrappers and stuff.
*/

#include "main.h"
#include "lodepng.h"
#include "lpi_screen.h"
#include "lpi_input.h"
#include "lpi_math3d.h"
#include "lpi_texture.h"
#include "lpi_text.h"

#include <vector>
#include <cmath>

using namespace lpi;

struct Deltaplex
{
  /// *** Code ***
  std::vector<unsigned char> code;
  int width;
  int height;
  
  void getInstruction(int& R, int& G, int& B, int& A)
  {
    R = code[4 * width * ip.y + 4 * ip.x + 0];
    G = code[4 * width * ip.y + 4 * ip.x + 1];
    B = code[4 * width * ip.y + 4 * ip.x + 2];
    A = code[4 * width * ip.y + 4 * ip.x + 3];
  }
  
  /// *** Stack ***
  std::vector<double> doubles;
  std::vector<int> ints;
  
  double popdouble()
  {
    if(doubles.size() == 0) error("pop from empty floating point stack");
    double result = doubles.back();
    doubles.pop_back();
    return result;
  }
  
  void pushdouble(double v)
  {
    doubles.push_back(v);
  }
  
  double& backdouble()
  {
    if(doubles.size() == 0) error("pop from empty floating point stack");
    return doubles.back();
  }
  
  int popint()
  {
    if(ints.size() == 0) error("pop from empty integer stack");
    int result = ints.back();
    ints.pop_back();
    return result;
  }
  
  void pushint(int v)
  {
    ints.push_back(v);
  }
  
  int& backint()
  {
    if(ints.size() == 0) error("pop from empty integer stack");
    return ints.back();
  }
  
  size_t intpointer;
  size_t floatpointer;
  
  //matrices and vectors from the floating point stack
  
  void popVector3(Vector3& out)
  {
    if(doubles.size() < 3) error("pop from empty floating point stack");
    out.z = popdouble();
    out.y = popdouble();
    out.x = popdouble();
  }
  
  void pushVector3(const Vector3& in)
  {
    pushdouble(in.x);
    pushdouble(in.y);
    pushdouble(in.z);
  }
  
  void popVector4(Vector4& out)
  {
    if(doubles.size() < 4) error("pop from empty floating point stack");
    out.w = popdouble();
    out.z = popdouble();
    out.y = popdouble();
    out.x = popdouble();
  }
  
  void pushVector4(const Vector4& in)
  {
    pushdouble(in.x);
    pushdouble(in.y);
    pushdouble(in.z);
    pushdouble(in.w);
  }
  
  void popMatrix3(Matrix3& out)
  {
    if(doubles.size() < 9) error("pop from empty floating point stack");
    for(int i = 8; i >= 0; i--) out.a[i] = popdouble();
  }
  
  void getMatrix3(Matrix3& out) //without popping
  {
    if(doubles.size() < 9) error("get from empty floating point stack");
    for(int i = 8; i >= 0; i--) out.a[i] = doubles[doubles.size() - 9 + i];
  }
  
  void pushMatrix3(const Matrix3& in)
  {
    for(int i = 0; i < 9; i++) pushdouble(in.a[i]);
  }
  
  void popMatrix4(Matrix4& out)
  {
    if(doubles.size() < 16) error("pop from empty floating point stack");
    for(int i = 15; i >= 0; i--) out.a[i] = popdouble();
  }
  
  void getMatrix4(Matrix4& out) //without popping
  {
    if(doubles.size() < 16) error("get from empty floating point stack");
    for(int i = 15; i >= 0; i--) out.a[i] = doubles[doubles.size() - 16 + i];
  }
  
  void pushMatrix4(const Matrix4& in)
  {
    for(int i = 0; i < 16; i++) pushdouble(in.a[i]);
  }
  
  /// *** Instruction Pointer ***
  struct IP
  {
    int x;
    int y;
    int dirx;
    int diry;
  };
  
  IP ip;
  
  void moveIP() //move the instruction pointer one step further
  {
    ip.x += ip.dirx;
    ip.y += ip.diry;
    if(ip.x >= width || ip.x < 0) ip.x = wrapmod(ip.x, width);
    if(ip.y >= width || ip.y < 0) ip.y = wrapmod(ip.y, height);
  }
  
  void moveIP(int i)
  {
    ip.x += i * ip.dirx;
    ip.y += i * ip.diry;
    if(ip.x >= width || ip.x < 0) ip.x = wrapmod(ip.x, width);
    if(ip.y >= width || ip.y < 0) ip.y = wrapmod(ip.y, height);
  }
  
  void setIR(int i) { code[4 * width * ip.y + 4 * ip.x + 0] = i; }
  void setIG(int i) { code[4 * width * ip.y + 4 * ip.x + 1] = i; }
  void setIB(int i) { code[4 * width * ip.y + 4 * ip.x + 2] = i; }
  void setIA(int i) { code[4 * width * ip.y + 4 * ip.x + 3] = i; }
  
  int getIR() { return code[4 * width * ip.y + 4 * ip.x + 0]; }
  int getIG() { return code[4 * width * ip.y + 4 * ip.x + 1]; }
  int getIB() { return code[4 * width * ip.y + 4 * ip.x + 2]; }
  int getIA() { return code[4 * width * ip.y + 4 * ip.x + 3]; }
  
  /// *** Memory Pointer ***
  IP mp;
  
  void moveMP()
  {
    mp.x += mp.dirx;
    mp.y += mp.diry;
    if(mp.x >= width || mp.x < 0) mp.x = wrapmod(mp.x, width);
    if(mp.y >= width || mp.y < 0) mp.y = wrapmod(mp.y, height);
  }
  
  void moveMP(int i)
  {
    mp.x += i * mp.dirx;
    mp.y += i * mp.diry;
    if(mp.x >= width || mp.x < 0) mp.x = wrapmod(mp.x, width);
    if(mp.y >= width || mp.y < 0) mp.y = wrapmod(mp.y, height);
  }
  
  //get/set memory pointer colors
  void setMR(int i) { code[4 * width * mp.y + 4 * mp.x + 0] = i; }
  void setMG(int i) { code[4 * width * mp.y + 4 * mp.x + 1] = i; }
  void setMB(int i) { code[4 * width * mp.y + 4 * mp.x + 2] = i; }
  void setMA(int i) { code[4 * width * mp.y + 4 * mp.x + 3] = i; }
  void setMC(int i, int c) { code[4 * width * mp.y + 4 * mp.x + c] = i; }
  
  int getMR() { return code[4 * width * mp.y + 4 * mp.x + 0]; }
  int getMG() { return code[4 * width * mp.y + 4 * mp.x + 1]; }
  int getMB() { return code[4 * width * mp.y + 4 * mp.x + 2]; }
  int getMA() { return code[4 * width * mp.y + 4 * mp.x + 3]; }
  int getMC(int c) { return code[4 * width * mp.y + 4 * mp.x + c]; }
  
  /// *** Gosub Stack ***
  std::vector<IP> gosubs;
  
  void popgosub()
  {
    if(gosubs.size() == 0) error("pop from empty gosub stack");
    ip = gosubs.back();
    gosubs.pop_back();
  }
  
  void pushgosub()
  {
    gosubs.push_back(ip);
  }
  
  /// *** Textures ***
  Texture textures[256];
  size_t current_texture;
  Texture& getTexture()
  {
    return textures[current_texture];
  }
  
  /// *** Strings ***
  std::string strings[256];
  size_t current_string;
  std::string& getString()
  {
    return strings[current_string];
  }
  
  /// *** Colors ***
  ColorRGB foreground;
  ColorRGB background;
  
  /// *** Time Anchor ***
  Uint32 anchorTime; //in milliseconds
  Uint32 lastAnchor;
  
  /// *** Near and Far clipping planes
  double near;
  double far;
  
  /// *** Ticks of "ticks since previous time you called this" command
  Uint32 lastTicks;
  
  /// === Functions ===
  
  void init()
  {
    ip.x = 0;
    ip.y = 0;
    ip.dirx = 1;
    ip.diry = 0;
    
    mp.x = 0;
    mp.y = 0;
    mp.dirx = 1;
    mp.diry = 0;
    
    anchorTime = 0;
    lastAnchor = 0;
    
    foreground = RGB_White;
    background = RGB_Black;
    
    glClearColor(0.0, 0.0, 0.0, 0.0);
    glColor4d(1.0, 1.0, 1.0, 1.0);
    glMatrixMode(GL_MODELVIEW);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_TEXTURE_2D);
    
    doubles.clear();
    ints.clear();
    gosubs.clear();
    
    width = 0;
    height = 0;
    
    floatpointer = 0;
    intpointer = 0;
    
    current_string = 0;
    current_texture = 0;
    
    near = 0.1;
    far = 10000.0;
    
    lastTicks = getTicks();
    
  }
  
  Deltaplex()
  {
    init();
  }
  
  void error(std::string text)
  {
    std::cout << "error: " << text << "\n";
    std::cout << "Program State:\n";
    std::cout << "x: " << ip.x << " y: " << ip.y << " dirx: " << ip.dirx << " diry: " << ip.diry << "\n";
    end();
  }
  
  void invalid()
  {
    error("invalid command");
  }
  
  void readPNG(std::string filename)
  {
    LodePNG::Decoder png_decoder;
    std::vector<unsigned char> buffer; //this will contain the file
    LodePNG::loadFile(buffer, filename); //load the image file with given filename
    png_decoder.decode(code, buffer); //decode the PNG
    if(png_decoder.hasError()) error("PNG error: " + png_decoder.getError());
    width = png_decoder.getWidth();
    height = png_decoder.getHeight();
  }
};



////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//############################################################################//
//############################################################################//
//############################################################################//
//############################################################################//
//############################################################################//
//############################################################################//
//############################################################################//
//############################################################################//
//############################################################################//
//############################################################################//
//############################################################################//
//############################################################################//
//############################################################################//
//############################################################################//
//############################################################################//
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////


int main(int argc, char *argv[])
{
  Deltaplex p; //the program
  
  int R, G, B, A; //the last read code pixels
  
  std::cout << "loading...\n";
  
  p.readPNG(argc > 1 ? argv[1] : "code.png");
  
  int counter = 0;
  
  std::cout << "done.\n";
  while(!done())
  {
    counter++;
    p.getInstruction(R, G, B, A);
    
    if(R < 128)
    {
      if(R < 64)
      {
        if(R == 0)
        {
        //"dark" NOP
        }
        //time
        else if(R == 10)
        {
          if(G == 0)
          {
            p.pushint(getTicks());
          }
          else if(G == 128)
          {
            int ticks = getTicks();
            p.pushint(ticks - p.lastTicks);
            p.lastTicks = ticks;
          }
          else if(G == 255)
          {
            SDL_Delay(p.popint());
          }
        }
        else if(R == 11)
        {
          if(G == 255)
          {
            while(getTicks() - p.lastAnchor < p.anchorTime)
            {
                SDL_Delay(1);
                if(done()) end();
            }
            p.lastAnchor = getTicks();
            break;
          }
        }
        else if(R == 12)
        {
          if(G == 0) p.anchorTime = p.popint();
          else if(G == 1) p.pushint(p.anchorTime);
        }
        else if(R == 13)
        {
          SDL_Delay(256 * G + B);
        }
        //integer stack
        else if(R == 20)
        {
          if     (G ==  0) {                     for(int i = 0; i < B; i++) p.popint(); }
          else if(G ==  1) { int N = p.popint(); for(int i = 0; i < N; i++) p.popint(); }
          else if(G == 40)
          {
            std::vector<int> values;
            values.resize(B);
            for(int j = 0; j < B; j++) values[B - 1 - j] = p.popint();
            for(int j = 0; j < B; j++) p.pushint(values[j]);
            for(int j = 0; j < B; j++) p.pushint(values[j]);
          }
          else if(G == 41)
          {
            int N = p.popint();
            std::vector<int> values;
            values.resize(N);
            for(int j = 0; j < N; j++) values[N - 1 - j] = p.popint();
            for(int j = 0; j < N; j++) p.pushint(values[j]);
            for(int j = 0; j < N; j++) p.pushint(values[j]);
          }
          else if(G == 81)
          {
            int M = p.popint();
            int N = p.popint();
            std::vector<int> temp;
            temp.insert(temp.begin(), p.ints.end() - N, p.ints.end());
            p.ints.erase(p.ints.end() - N, p.ints.end());
            p.ints.insert(p.ints.end() - M, temp.begin(), temp.end());
          }
          else if(G == 120)
          {
            for(int i = 0; i < B / 2; i++)
            {
              int temp = p.ints[p.ints.size() - 1 - i];
              p.ints[p.ints.size() - 1 - i] = p.ints[p.ints.size() - B + i];
              p.ints[p.ints.size() - B + i] = temp;
            }
          }
          else if(G == 121)
          {
            for(int i = 0; i < B / 2; i++)
            {
              int N = p.popint();
              int temp = p.ints[p.ints.size() - 1 - i];
              p.ints[p.ints.size() - 1 - i] = p.ints[p.ints.size() - B + i];
              p.ints[p.ints.size() - N + i] = temp;
            }
          }
        }
        else if(R == 21)
        {
          std::vector<int> temp;
          temp.insert(temp.begin(), p.ints.end() - G, p.ints.end());
          p.ints.erase(p.ints.end() - G, p.ints.end());
          p.ints.insert(p.ints.end() - B, temp.begin(), temp.end());
        }
        else if(R == 22)
        {
          if(G == 0) { p.intpointer = p.popint(); }
          else if(G == 128) { p.ints[p.intpointer] = p.ints.back(); }
          else if(G == 129) { p.pushint(p.ints[p.intpointer]); }
        }
        //floating point stack
        else if(R == 25)
        {
          if(G == 0)      {                     for(int i = 0; i < B; i++) p.popdouble(); }
          else if(G == 1) { int N = p.popint(); for(int i = 0; i < N; i++) p.popdouble(); }
          else if(G == 40)
          {
            std::vector<double> values;
            values.resize(B);
            for(int j = 0; j < B; j++) values[B - 1 - j] = p.popdouble();
            for(int j = 0; j < B; j++) p.pushdouble(values[j]);
            for(int j = 0; j < B; j++) p.pushdouble(values[j]);
          }
          else if(G == 41)
          {
            int N = p.popint();
            std::vector<double> values;
            values.resize(N);
            for(int j = 0; j < N; j++) values[N - 1 - j] = p.popdouble();
            for(int j = 0; j < N; j++) p.pushdouble(values[j]);
            for(int j = 0; j < N; j++) p.pushdouble(values[j]);
          }
          else if(G == 81)
          {
            int M = p.popint();
            int N = p.popint();
            std::vector<double> temp;
            temp.insert(temp.begin(), p.doubles.end() - N, p.doubles.end());
            p.doubles.erase(p.doubles.end() - N, p.doubles.end());
            p.doubles.insert(p.doubles.end() - M, temp.begin(), temp.end());
          }
          else if(G == 120)
          {
            for(int i = 0; i < B / 2; i++)
            {
              double temp = p.doubles[p.doubles.size() - 1 - i];
              p.doubles[p.doubles.size() - 1 - i] = p.doubles[p.doubles.size() - B + i];
              p.doubles[p.doubles.size() - B + i] = temp;
            }
          }
          else if(G == 121)
          {
            for(int i = 0; i < B / 2; i++)
            {
              int N = p.popint();
              double temp = p.doubles[p.doubles.size() - 1 - i];
              p.doubles[p.doubles.size() - 1 - i] = p.doubles[p.doubles.size() - B + i];
              p.doubles[p.doubles.size() - N + i] = temp;
            }
          }
          else if(G == 160) { p.pushdouble(p.doubles[p.floatpointer]); }
          else if(G == 161) { p.doubles[p.floatpointer] = p.doubles.back(); }
          else if(G == 162) { p.floatpointer = p.popint(); }
        }
        else if(R == 26)
        {
          std::vector<double> temp;
          temp.insert(temp.begin(), p.doubles.end() - G, p.doubles.end());
          p.doubles.erase(p.doubles.end() - G, p.doubles.end());
          p.doubles.insert(p.doubles.end() - B, temp.begin(), temp.end());
        }
        else if(R == 27)
        {
          if(G == 0) { p.floatpointer = p.popint(); }
          else if(G == 128) { p.pushdouble(p.doubles[p.floatpointer]); }
          else if(G == 129) { p.doubles[p.floatpointer] = p.doubles.back(); }
        }
        //swapping between int and float stack
        else if(R == 28)
        {
          if(B == 128)
          {
            if(G == 0) { int val = p.popint(); p.pushdouble(val); }
            else if(G == 1) { double val = p.popdouble(); p.pushint(int(val)); }
            else if(G == 2)
            {
              int N = p.popint();
              for(int i = 0; i < N; i++) p.pushdouble(p.ints[p.ints.size() - N + i]);
              p.ints.erase(p.ints.end() - N, p.ints.end());
            }
            else if(G == 3)
            {
              int N = p.popint();
              for(int i = 0; i < N; i++) p.pushint(int(p.doubles[p.doubles.size() - N + i]));
              p.doubles.erase(p.doubles.end() - N, p.doubles.end());
            }
          }
        }
        // //screen
        else if(R == 40)
        {
          int M = p.popint();
          int N = p.popint();
          screen(N, M, 0, "Deltaplex");
        }
        else if(R == 41)
        {
          if(G == 0) redraw();
          else if(G == 128) glClearColor(p.background.r / 255.0, p.background.g / 255.0, p.background.b / 255.0, p.background.a / 255.0);
          else if(G == 129) glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
        }
        // //2D graphics
        else if(R == 50) set2DScreen();
        else if(R == 51)
        {
          std::string text;
          if(B == 0) text = p.getString();
          else if(B == 1) text = valtostr(p.popint());
          else if(B == 2) text = valtostr(p.popdouble());
          
          Markup markup(p.foreground, p.background, 1);
          
          if(G == 0) markup.nomarkup = true;
          
          int y = p.popint();
          int x = p.popint();
          
          print(text, x, y, markup);
        }
        else if(R == 52)
        {
          int y = p.popint();
          int x = p.popint();
          
          p.getTexture().draw(x, y);
        }
        //3D graphics & OpenGL
        else if(R == 60)
        {
          if(G == 0) set3DScreen(p.near, p.far);
          else if(G == 1)
          {
            p.far = p.popdouble();
            p.near = p.popdouble();
          }
        }
        else if(R == 61)
        {
          switch(G)
          {
            case 0: glBegin(GL_POINTS); break;
            case 1: glBegin(GL_LINES); break;
            case 2: glBegin(GL_LINE_LOOP); break;
            case 3: glBegin(GL_LINE_STRIP); break;
            case 4: glBegin(GL_TRIANGLES); break;
            case 5: glBegin(GL_TRIANGLE_STRIP); break;
            case 6: glBegin(GL_TRIANGLE_FAN); break;
            case 7: glBegin(GL_QUADS); break;
            case 8: glBegin(GL_QUAD_STRIP); break;
            case 9: glBegin(GL_POLYGON); break;
            case 255: glEnd(); break;
            default: p.error("invalid OpenGL primitive");
          }
        }
        else if(R == 62)
        {
          if(G == 0)
          {
            if(B == 0) glMatrixMode(GL_MODELVIEW);
            else if(G == 1) glMatrixMode(GL_PROJECTION);
          }
          else if(G == 1) glLoadIdentity();
          else if(G == 2)
          {
            double a[16]; for(int i = 15; i >= 0; i--) a[i] = p.popdouble();
            glLoadMatrixd(a);
          }
          else if(G == 3)
          {
            double a[16]; for(int i = 15; i >= 0; i--) a[i] = p.popdouble();
            glMultMatrixd(a);
          }
          else if(G == 40) glPushMatrix();
          else if(G == 41) glPopMatrix();
          else if(G == 200)
          {
            if(B == 0)
            {
              double a[16];
              glGetDoublev(GL_MODELVIEW_MATRIX, a);
              for(int i = 0; i < 16; i++) p.pushdouble(a[i]);
            }
            else if(B == 1)
            {
            double a[16];
            glGetDoublev(GL_PROJECTION_MATRIX, a);
            for(int i = 0; i < 16; i++) p.pushdouble(a[i]);
            }
          }
        }
        else if(R == 63)
        {
          if(G == 0 && B == 2)
          {
            double y = p.popdouble();
            double x = p.popdouble();
            glVertex2d(x, y);
          }
          else if(G == 0 && B == 3)
          {
            double z = p.popdouble();
            double y = p.popdouble();
            double x = p.popdouble();
            
            glVertex3d(x, y, z);
          }
          else if(G == 64 && B == 2)
          {
            double y = p.popdouble();
            double x = p.popdouble();
            glTexCoord2d(x, y);
          }
          else if(G == 128 && B == 4)
          {
            glColor4i(p.foreground.r, p.foreground.g, p.foreground.b, p.foreground.a);
          }
          else if(G == 129 && B == 4)
          {
            int a = p.popint();
            int b = p.popint();
            int g = p.popint();
            int r = p.popint();
            glColor4i(r, g, b, a);
          }
          else if(G == 130 && B == 4)
          {
            double a = p.popdouble();
            double b = p.popdouble();
            double g = p.popdouble();
            double r = p.popdouble();
            glColor4d(r, g, b, a);
          }
        }
        else
        {
          p.invalid();
          //nothing
        }
      }
      else //R >= 64
      {
        if(R == 64)
        {
          if(G == 0 && B == 0) glEnable(GL_TEXTURE_2D);
          else if(G == 0 && B == 1) glEnable(GL_DEPTH_TEST);
          else if(G == 1 && B == 0) glDisable(GL_TEXTURE_2D);
          else if(G == 1 && B == 1) glDisable(GL_DEPTH_TEST);
        }
        else if(R == 65)
        {
          if(G == 0)
          {
            double z = p.popdouble();
            double y = p.popdouble();
            double x = p.popdouble();
            glTranslated(x, y, z);
          }
          else if(G == 1)
          {
            double z = p.popdouble();
            double y = p.popdouble();
            double x = p.popdouble();
            double angle = p.popdouble();
            glRotated(angle, x, y, z);
          }
          else if(G == 2)
          {
            double z = p.popdouble();
            double y = p.popdouble();
            double x = p.popdouble();
            glScaled(x, y, z);
          }
        }
        // //commands about getting/setting data from the pixel code
        else if(R == 100)
        {
          if(G == 0)
          {
            p.mp.y = p.popint();
            p.mp.x = p.popint();
          }
          else if(G == 1)
          {
            p.mp.diry = p.popint();
            p.mp.dirx = p.popint();
          }
          else if(G == 100) { p.mp.x = p.ip.x; p.mp.y = p.ip.y; }
          else if(G == 101) { p.mp.dirx = p.ip.dirx; p.mp.diry = p.ip.diry; }
        }
        else if(R == 101) { p.mp.x = G; p.mp.y = B; }
        else if(R == 102) { p.mp.dirx = G; p.mp.diry = B; }
        else if(R == 103)
        {
          if(G == 0) p.moveMP();
          else if(G == 255) p.moveMP(-1);
        }
        else if(R == 104)
        {
          if(G == 0)
          {
            unsigned int i = (unsigned int)p.popint();
            p.setMR((i >> 24) & 255);
            p.setMG((i >> 16) & 255);
            p.setMB((i >>  8) & 255);
            p.setMA((i >>  0) & 255);
          }
          else if(G == 1) p.pushint(16777216 * p.getMR() + 65536 * p.getMG() + 256 * p.getMB() + p.getMA());
          else if(G == 100)
          {
            //this only works properly if both float and int are 32-bit
            float f = float(p.popdouble());
            unsigned int* ip = reinterpret_cast<unsigned int*>(&f);
            unsigned int i = *ip;
            p.setMR((i >> 24) & 255);
            p.setMG((i >> 16) & 255);
            p.setMB((i >>  8) & 255);
            p.setMA((i >>  0) & 255);
          }
          else if(G == 101)
          {
            unsigned int i = 16777216 * p.getMR() + 65536 * p.getMG() + 256 * p.getMB() + p.getMA();
            float* fp = reinterpret_cast<float*>(&i);
            float f = *fp;
            p.pushdouble(f);
          }
        }
        else if(R == 105)
        {
          if(G == 0) p.setMR(p.popint());
          else if(G == 1) p.setMG(p.popint());
          else if(G == 2) p.setMB(p.popint());
          else if(G == 3) p.setMA(p.popint());
          else if(G == 4) p.pushint(p.getMR());
          else if(G == 5) p.pushint(p.getMG());
          else if(G == 6) p.pushint(p.getMB());
          else if(G == 7) p.pushint(p.getMA());
        }
        else if(R == 106)
        {
          int i = 256 * G + B;
          i++;
          p.setIG((i >> 8) & 255);
          p.setIB(i & 255);
        }
        else if(R == 107)
        {
          int i = 256 * G + B;
          if(i > 0)
          {
            i--;
            p.setIG((i >> 8) & 255);
            p.setIB(i & 255);
          }
          else p.moveIP();
        }
        else if(R == 108)
        {
          if(G == 0) p.pushint(p.width);
          else if(G == 1) p.pushint(p.height);
        }
        // //mouse & keyboard
        else if(R == 120)
        {
          if(G == 0) p.pushint(globalLMB);
          else if(G == 1) p.pushint(globalRMB);
          else if(G == 2) p.pushint(mouseWheelUp);
          else if(G == 3) p.pushint(mouseWheelDown);
          else if(G == 40) p.pushint(mouseX);
          else if(G == 41) p.pushint(mouseY);
        }
        else if(R == 121)
        {
          if(G == 0) p.pushint(inkeys[B]);
          else if(G == 1) { p.pushint(inkeys[p.popint()]); }
        }
        else
        {
          p.invalid();
          //nothing
        }
      }
    }
    else //R >= 128
    {
      if(R < 192)
      {
        //"medium" NOP
        if(R == 128) { }
    // //color
        else if(R == 130)
        {
          if(G == 0)
          {
            int r, g, b, a;
            p.moveIP();
            p.getInstruction(r, g, b, a); //getting color instead of instruction in this case
            p.foreground.r = r;
            p.foreground.g = g;
            p.foreground.b = b;
            p.foreground.a = a;
          }
          else if(G == 0)
          {
            int r, g, b, a;
            p.moveIP();
            p.getInstruction(r, g, b, a); //getting color instead of instruction in this case
            p.background.r = r;
            p.background.g = g;
            p.background.b = b;
            p.background.a = a;
          }
        }
        else if(R == 131)
        {
          if(B == 0)
          {
            if(G == 0) { p.foreground.a = p.popint(); p.foreground.b = p.popint(); p.foreground.g = p.popint(); p.foreground.r = p.popint(); }
            else if(G == 1) { p.background.a = p.popint(); p.background.b = p.popint(); p.background.g = p.popint(); p.background.r = p.popint(); }
          }
          else if(B == 1)
          {
            if(G == 0) { p.foreground.a = int(255 * p.popdouble()); p.foreground.b = int(255 * p.popdouble()); p.foreground.g = int(255 * p.popdouble()); p.foreground.r = int(255 * p.popdouble()); }
            else if(G == 1) { p.background.a = int(255 * p.popdouble()); p.background.b = int(255 * p.popdouble()); p.background.g = int(255 * p.popdouble()); p.background.r = int(255 * p.popdouble()); }
          }
        }
        else if(R == 132)
        {
          if(B == 0)
          {
            if(G == 0) { p.pushint(p.foreground.r); p.pushint(p.foreground.g); p.pushint(p.foreground.b); p.pushint(p.foreground.a); }
            else if(G == 1) { p.pushint(p.background.r); p.pushint(p.background.g); p.pushint(p.background.b); p.pushint(p.background.a); }
          }
          else if(B == 1)
          {
            if(G == 0) { p.pushdouble(p.foreground.r / 255.0); p.pushdouble(p.foreground.g / 255.0); p.pushdouble(p.foreground.b / 255.0); p.pushdouble(p.foreground.a / 255.0);}
            else if(G == 1) { p.pushdouble(p.background.r / 255.0); p.pushdouble(p.background.g / 255.0); p.pushdouble(p.background.b / 255.0); p.pushdouble(p.background.a / 255.0);}
          }
        }
        else if(R == 133)
        {
          ColorRGB temp = p.foreground;
          p.foreground = p.background;
          p.background = temp;
        }
        // //text
        else if(R == 140)
        {
          p.current_string = p.popint();
        }
        else if(R == 141)
        {
          if(G == 0) p.getString().clear();
          else if(G == 128)
          {
            if(B > 3) p.error("invalid color channel");
            std::string s;
            bool stop = false;
            while(!stop)
            {
              int c = p.getMC(B);
              p.moveMP();
              if(c == 0) stop = true;
              else  s.push_back(c);
            }
            p.getString() += s;
          }
          else if(G == 129)
          {
            if(B > 3) p.error("invalid color channel");
            std::string s;
            int amount = p.popint();
            for(int i = 0; i < amount; i++)
            {
              int c = p.getMC(B);
              p.moveMP();
              s.push_back(c);
            }
            p.getString() += s;
          }
          else if(G == 130)
          {
            std::string s = valtostr(p.popint());
            p.getString() += s;
          }
          else if(G == 131)
          {
            std::string s = valtostr(p.popdouble());
            p.getString() += s;
          }
        }
        else if(R == 142)
        {
          p.pushint(p.getString().length());
        }
        
        // //integer math
        else if(R == 160)
        {
          int y = p.popint();
          int x = p.popint();
          switch(G)
          {
            case 0: p.pushint(x  + y); break;
            case 1: p.pushint(x  - y); break;
            case 2: p.pushint(x  * y); break;
            case 3: p.pushint(x  / y); break;
            case 4: p.pushint(x  % y); break;
            case 40: p.pushint(x == y); break;
            case 41: p.pushint(x != y); break;
            case 42: p.pushint(x  > y); break;
            case 43: p.pushint(x  < y); break;
            case 44: p.pushint(x >= y); break;
            case 45: p.pushint(x <= y); break;
            case 80: p.pushint(x  & y); break;
            case 81: p.pushint(x  | y); break;
            case 82: p.pushint(x  ^ y); break;
            case 120: p.pushint(x && y); break;
            case 121: p.pushint(x || y); break;
          }
        }
        else if(R == 161)
        {
          switch(G)
          {
            case 0: p.backint() = -p.backint(); break;
            case 40: p.backint() = ~p.backint(); break;
            case 41: p.backint() = !p.backint(); break;
            case 80: p.backint()++; break;
            case 81: p.backint()--; break;
            case 120: p.backint() = template_abs(p.backint()); break;
            case 200: p.backint() = p.backint() * p.backint(); break;
          }
        }
        // //floating point math
        else if(R == 170)
        {
          double y = p.popdouble();
          double x = p.popdouble();
          switch(G)
          {
            case 0: p.pushdouble(x  + y); break;
            case 1: p.pushdouble(x  - y); break;
            case 2: p.pushdouble(x  * y); break;
            case 3: p.pushdouble(x  / y); break;
            case 40: p.pushint(x == y); break;
            case 41: p.pushint(x != y); break;
            case 42: p.pushint(x  > y); break;
            case 43: p.pushint(x  < y); break;
            case 44: p.pushint(x >= y); break;
            case 45: p.pushint(x <= y); break;
            case 160: p.pushdouble(std::pow(x, y)); break;
            case 161: p.pushdouble(std::atan2(x, y)); break;
          }
        }
        else if(R == 171)
        {
          switch(G)
          {
            case 0: p.backdouble() = -p.backdouble(); break;
            case 1: p.backdouble() = 1.0 / p.backdouble(); break;
            case 80: p.backdouble()++; break;
            case 81: p.backdouble()--; break;
            case 120: p.backdouble() = template_abs(p.backdouble()); break;
            case 121: p.backdouble() = std::floor(p.backdouble()); break;
            case 122: p.backdouble() = std::ceil(p.backdouble()); break;
            case 200: p.backdouble() = p.backdouble() * p.backdouble(); break;
            case 201: p.backdouble() = std::sqrt(p.backdouble()); break;
            case 202: p.backdouble() = std::log(p.backdouble()); break;
            case 203: p.backdouble() = std::exp(p.backdouble()); break;
            case 255: p.backdouble() = 3.141592653589793116 * 2 * p.backdouble(); break;
          }
        }
        else if(R == 172)
        {
          switch(G)
          {
            case 0: p.backdouble() = std::sin(p.backdouble()); break;
            case 1: p.backdouble() = std::cos(p.backdouble()); break;
            case 2: p.backdouble() = std::tan(p.backdouble()); break;
            case 6: p.backdouble() = std::asin(p.backdouble()); break;
            case 7: p.backdouble() = std::acos(p.backdouble()); break;
            case 8: p.backdouble() = std::atan(p.backdouble()); break;
          }
        }
        else if(R == 173)
        {
          switch(G)
          {
            case 0: p.pushdouble(3.141592653589793116); break;
            case 1: p.pushdouble(6.283185307179586232); break;
            case 2: p.pushdouble(2.718281828459045091); break;
          }
        }
        // //matrix/vector math
        else if(R == 175 || R == 176)
        {
          int amount;
          if(R == 175) amount = B;
          else if(R == 176) amount = p.popint();
          
          switch(G)
          {
            if(G == 0) { for(int i = 0; i < amount; i++) p.doubles[p.doubles.size() - amount + i] = -p.doubles[p.doubles.size() - amount + i]; }
            else if(G == 1)
            {
              std::vector<double> temp1(amount);
              std::vector<double> temp2(amount);
              for(int i = 0; i < amount; i++) temp1[i] = p.popdouble();
              for(int i = 0; i < amount; i++) temp2[i] = p.popdouble();
              for(int i = 0; i < amount; i++) p.pushdouble(temp1[amount - 1 - i] + temp2[amount - 1 - i]);
            }
            else if(G == 2)
            {
              std::vector<double> temp1(amount);
              std::vector<double> temp2(amount);
              for(int i = 0; i < amount; i++) temp1[i] = p.popdouble();
              for(int i = 0; i < amount; i++) temp2[i] = p.popdouble();
              for(int i = 0; i < amount; i++) p.pushdouble(temp1[amount - 1 - i] - temp2[amount - 1 - i]);
            }
            else if(G == 3)
            {
              std::vector<double> temp1(amount);
              std::vector<double> temp2(amount);
              for(int i = 0; i < amount; i++) temp1[i] = p.popdouble();
              for(int i = 0; i < amount; i++) temp2[i] = p.popdouble();
              for(int i = 0; i < amount; i++) p.pushdouble(temp1[amount - 1 - i] * temp2[amount - 1 - i]);
            }
            else if(G == 4)
            {
              std::vector<double> temp1(amount);
              std::vector<double> temp2(amount);
              for(int i = 0; i < amount; i++) temp1[i] = p.popdouble();
              for(int i = 0; i < amount; i++) temp2[i] = p.popdouble();
              for(int i = 0; i < amount; i++) p.pushdouble(temp1[amount - 1 - i] / temp2[amount - 1 - i]);
            }
            else if(G == 5)
            {
              std::vector<double> temp1(amount);
              std::vector<double> temp2(amount);
              int equal = 1;
              for(int i = 0; i < amount; i++) temp1[i] = p.popdouble();
              for(int i = 0; i < amount; i++) temp2[i] = p.popdouble();
              for(int i = 0; i < amount; i++) if(temp1[i] != temp2[i]) equal = 0;
              p.pushint(equal);
            }
            else if(G == 5)
            {
              std::vector<double> temp1(amount);
              std::vector<double> temp2(amount);
              int notequal = 0;
              for(int i = 0; i < amount; i++) temp1[i] = p.popdouble();
              for(int i = 0; i < amount; i++) temp2[i] = p.popdouble();
              for(int i = 0; i < amount; i++) if(temp1[i] != temp2[i]) notequal = 1;
              p.pushint(notequal);
            }
            else if(G == 40)
            {
              double d = p.popdouble();
              for(int i = 0; i < amount; i++) p.doubles[p.doubles.size() - amount + i] = d * p.doubles[p.doubles.size() - amount + i];
            }
          }
        }
        else if(R == 180)
        {
          if(B == 3)
          {
            if(G == 0)
            {
              Vector3 v; p.popVector3(v);
              p.pushdouble(length(v));
            }
            else if(G == 1)
            {
              Vector3 v; p.popVector3(v);
              p.pushdouble(lengthsq(v));
            }
          }
        }
        else if(R == 181)
        {
          if(B == 3)
          {
            if(G == 0)
            {
              Vector3 w; p.popVector3(w);
              Vector3 v; p.popVector3(v);
              p.pushdouble(dot(v, w));
            }
            else if(G == 1)
            {
              Vector3 w; p.popVector3(w);
              Vector3 v; p.popVector3(v);
              p.pushVector3(cross(v, w));
            }
            else if(G == 40)
            {
              Vector3 v; p.popVector3(v);
              v.normalize();
              p.pushVector3(v);
            }
          }
        }
        else if(R == 182)
        {
          if(B == 3)
          {
            if(G == 0)
            {
              Matrix3 m; p.popMatrix3(m);
              m.invert();
              p.pushMatrix3(m);
            }
            else if(G == 1)
            {
              Matrix3 m; p.popMatrix3(m);
              p.pushdouble(m.determinant());
            }
            else if(G == 40)
            {
              Matrix3 b; p.popMatrix3(b);
              Matrix3 a; p.popMatrix3(a);
              p.pushMatrix3(a * b);
            }
            else if(G == 80)
            {
              Vector3 b; p.popVector3(b);
              Matrix3 a; p.popMatrix3(a);
              p.pushVector3(a * b);
            }
          }
          else if(B == 4)
          {
            if(G == 0)
            {
              Matrix4 m; p.popMatrix4(m);
              m.invert();
              p.pushMatrix4(m);
            }
            else if(G == 1)
            {
              Matrix4 m; p.popMatrix4(m);
              p.pushdouble(m.determinant());
            }
            else if(G == 40)
            {
              Matrix4 b; p.popMatrix4(b);
              Matrix4 a; p.popMatrix4(a);
              p.pushMatrix4(a * b);
            }
            else if(G == 80)
            {
              Vector4 b; p.popVector4(b);
              Matrix4 a; p.popMatrix4(a);
              p.pushVector4(a * b);
            }
            else if(G == 81)
            {
              Vector3 b; p.popVector3(b);
              Matrix4 a; p.popMatrix4(a);
              p.pushVector3(a * b);
            }
          }
        }
        else if(R == 183)
        {
          if(B == 3)
          {
            if(G == 0) for(int i = 0; i < 9; i++) p.pushdouble(0.0);
            else if(G == 1) for(int i = 0; i < 9; i++) p.pushdouble(i % 3 == i / 3 ? 1.0 : 0.0);
            else if(G == 40) for(int i = 0; i < 3; i++) p.pushdouble(0.0);
            else if(G == 41) { p.pushdouble(1.0); p.pushdouble(0.0); p.pushdouble(0.0); }
            else if(G == 42) { p.pushdouble(0.0); p.pushdouble(1.0); p.pushdouble(0.0); }
            else if(G == 43) { p.pushdouble(0.0); p.pushdouble(0.0); p.pushdouble(1.0); }
            
          }
          else if(B == 4)
          {
            if(G == 0) for(int i = 0; i < 16; i++) p.pushdouble(0.0);
            else if(G == 1) for(int i = 0; i < 16; i++) p.pushdouble(i % 4 == i / 4 ? 1.0 : 0.0);
            else if(G == 40) for(int i = 0; i < 4; i++) p.pushdouble(0.0);
            else if(G == 41) { p.pushdouble(1.0); p.pushdouble(0.0); p.pushdouble(0.0); p.pushdouble(1.0); }
            else if(G == 42) { p.pushdouble(0.0); p.pushdouble(1.0); p.pushdouble(0.0); p.pushdouble(1.0); }
            else if(G == 43) { p.pushdouble(0.0); p.pushdouble(0.0); p.pushdouble(1.0); p.pushdouble(1.0); }
          }
        }
        else if(R == 184)
        {
          if(G == 3 && B == 4) p.pushdouble(1.0);
          else if(G == 4 && B == 3)
          {
            Vector4 v; p.popVector4(v);
            Vector3 w; v.convertTo(w);
            p.pushVector3(w);
          }
        }
        // //camera matrix transformations
        else if(R == 190)
        {
          if(B == 4)
          {
            if(G == 0)
            {
              Vector3 d; p.popVector3(d);
              Matrix4 m;
              getTranslationMatrix4(m, d);
              p.pushMatrix4(m);
            }
            else if(G == 1)
            {
              double angle = p.popdouble();
              Vector3 axis; p.popVector3(axis);
              Matrix4 m;
              getRotationMatrix4(m, axis, angle);
              p.pushMatrix4(m);
            }
          }
        }
        else if(R == 191)
        {
          if(B == 3)
          {
            if(G == 0) //translate 3x3+3 camera
            {
              Vector3 d; p.popVector3(d);
              
              Vector3 pos; p.popVector3(pos); Vector3 dir; p.popVector3(dir); Vector3 v; p.popVector3(v); Vector3 u; p.popVector3(u);
              Transformation3 t(pos.x, pos.y, pos.z, u.x,   u.y,   u.z, v.x,   v.y,   v.z, dir.x, dir.y, dir.z);

              
              t.move(d);
              
              p.pushVector3(t.getU()); p.pushVector3(t.getV()); p.pushVector3(t.getDir()); p.pushVector3(t.getPos());
            }
            else if(G == 1)
            {
              double angle = p.popdouble();
              Vector3 axis; p.popVector3(axis);
              
              Vector3 pos; p.popVector3(pos); Vector3 dir; p.popVector3(dir); Vector3 v; p.popVector3(v); Vector3 u; p.popVector3(u);
              Transformation3 t(pos.x, pos.y, pos.z, u.x,   u.y,   u.z, v.x,   v.y,   v.z, dir.x, dir.y, dir.z);

              
              t.rotate(axis, angle);
              
              p.pushVector3(t.getU()); p.pushVector3(t.getV()); p.pushVector3(t.getDir()); p.pushVector3(t.getPos());
            }
            else if(G == 128) //lookat
            {
              Vector3 target; p.popVector3(target);
              
              Vector3 pos; p.popVector3(pos); Vector3 dir; p.popVector3(dir); Vector3 v; p.popVector3(v); Vector3 u; p.popVector3(u);
              Transformation3 t(pos.x, pos.y, pos.z, u.x,   u.y,   u.z, v.x,   v.y,   v.z, dir.x, dir.y, dir.z);
              
              t.lookAt(target);
              
              p.pushVector3(t.getU()); p.pushVector3(t.getV()); p.pushVector3(t.getDir()); p.pushVector3(t.getPos());
            }
            else if(G >= 191 && G <= 196)
            {
              double angle = p.popdouble();
              
              Vector3 pos; p.popVector3(pos); Vector3 dir; p.popVector3(dir); Vector3 v; p.popVector3(v); Vector3 u; p.popVector3(u);
              Transformation3 t(pos.x, pos.y, pos.z, u.x,   u.y,   u.z, v.x,   v.y,   v.z, dir.x, dir.y, dir.z);
              
              switch(G)
              {
                case 191: t.pitch(angle); break;
                case 192: t.yawSpace(angle); break;
                case 193: t.roll(angle); break;
                case 194: t.rotate(Vector3(1,0,0), angle); break;
                case 195: t.rotate(Vector3(0,1,0), angle); break;
                case 196: t.rotate(Vector3(0,0,1), angle); break;
              }
              
              p.pushVector3(t.getU()); p.pushVector3(t.getV()); p.pushVector3(t.getDir()); p.pushVector3(t.getPos());
            }
            
            else if(G == 254)
            {
              Transformation3 t(0, 0, 0,
                                1, 0, 0,
                                0, 1, 0,
                                0, 0, 1);
              
              p.pushVector3(t.getU());
              p.pushVector3(t.getV());
              p.pushVector3(t.getDir());
              p.pushVector3(t.getPos());
            }
            else if(G == 255)
            {
              Vector3 pos; p.popVector3(pos);
              Matrix3 m; p.popMatrix3(m);
              
              /*this must:
              1) Invert the 3x3 matrix
              2) put it in a 4x4 matrix with the new rows 0 except the bottom right element 1
              3) glMultMatrixd this
              4) glTranslated over the pos vector*/

              m.invert();
              transformGLMatrix(m, pos);
            }
          }
          else p.invalid();
        }
        else
        {
          p.invalid();
          //nothing
        }
      }
      else //R >= 192
      {
        // //pushing numbers to the integer stack
        if(R == 200) { p.pushint(256 * G + B); }
        else if(R == 201) { p.pushint(p.popint() + 16777216 * G + 65536 * B); }
        else if(R == 202) { p.pushint(G); p.pushint(B); }
        else if(R == 204) { p.pushint(-256 * G - B); }
        else if(R == 205) { p.pushint(p.popint() - 16777216 * G - 65536 * B); }
        else if(R == 206) { p.pushint(int(char(G))); p.pushint(int(char(B))); }
        // //pushing numbers to the floating point stack
        else if(R == 210) { p.pushdouble(256 * G + B); }
        else if(R == 211) { p.pushdouble(p.popdouble() + 16777216.0 * G + 65536.0 * B); }
        else if(R == 212) { p.pushdouble(p.popdouble() + G / 256.0 + B / 65536.0); }
        else if(R == 213) { p.pushdouble(p.popdouble() + G / 16777216.0 + B / 4294967296.0 ); }
        else if(R == 214) { p.pushdouble(-256 * G - B); }
        else if(R == 215) { p.pushdouble(p.popdouble() - 16777216.0 * G - 65536.0 * B); }
        else if(R == 216) { p.pushdouble(p.popdouble() - G / 256.0 - B / 65536.0); }
        else if(R == 217) { p.pushdouble(p.popdouble() - G / 16777216.0 - B / 4294967296.0 ); }
        else if(R == 218) { p.pushdouble(G + B / 256.0); }
        else if(R == 219) { p.pushdouble(G - B / 256.0); }
        else if(R == 220) { p.pushdouble(G / 256.0 + B / 65536.0); }
        // //program flow
        else if(R == 230)
        {
          if(G == 0 && B == 0) p.ip.dirx = p.ip.diry = 0;
          else if(G == 0 && B == 255) { p.ip.dirx = 0; p.ip.diry = -1; }
          else if(G == 1 && B == 0) { p.ip.dirx = 1; p.ip.diry = 0; }
          else if(G == 0 && B == 1) { p.ip.dirx = 0; p.ip.diry = 1; }
          else if(G == 255 && B == 0) { p.ip.dirx = -1; p.ip.diry = 0; }
        }
        else if(R == 231)
        {
          if(G == 128) { int oldx = p.ip.dirx; int oldy = p.ip.diry; p.ip.dirx = -oldy; p.ip.diry = -oldx; }
          else if(G == 129) { int oldx = p.ip.dirx; int oldy = p.ip.diry; p.ip.dirx = oldy; p.ip.diry = oldx; }
        }
        else if(R == 232)
        {
          if(G == 0) { p.ip.y = wrapmod(p.popint(), p.height); p.ip.x = wrapmod(p.popint(), p.width); }
          else if(G == 1) { p.ip.x += p.popint(); p.ip.y += p.popint(); }
          else if(G == 2) { p.pushgosub(); p.ip.y = wrapmod(p.popint(), p.height); p.ip.x = wrapmod(p.popint(), p.width); }
          else if(G == 3) { p.pushgosub(); p.ip.x += p.popint(); p.ip.y += p.popint(); }
          else if(G == 255) { p.popgosub(); }
        }
        else if(R == 233)
        {
          if(G == 0) p.moveIP();
          else if(G == 1) { if(p.popint()) p.moveIP(); }
          else if(G == 2) { if(!p.popint()) p.moveIP(); }
        }
        else if(R == 234) sleep();
        // //texture
        else if(R == 240)
        {
          p.current_texture = p.popint();
        }
        else if(R == 241)
        {
          if(G == 0)
          {
            size_t y2 = size_t(p.popint());
            size_t x2 = size_t(p.popint());
            size_t y = size_t(p.popint());
            size_t x = size_t(p.popint());
            
            p.getTexture().create(&p.code[0], p.width, p.height, AE_Opaque, x, y, x2, y2);
          }
          else if(G == 1)
          {
            size_t h = p.popint();
            size_t w = p.popint();
            
            p.getTexture().create(p.foreground, w, h);
          }
          else if(G == 128) p.getTexture().bind();
          else if(G == 129) p.getTexture().upload();
        }
        else if(R == 242)
        {
          Texture& t = p.getTexture();
          int y = p.popint();
          int x = p.popint();
          
          switch(G)
          {
            case 0: p.pushint(t.getPixel(x, y, 0)); p.pushint(t.getPixel(x, y, 1)); p.pushint(t.getPixel(x, y, 2)); p.pushint(t.getPixel(x, y, 3)); break;
            case 1: p.pushdouble(t.getPixel(x, y, 0) / 255.0); p.pushdouble(t.getPixel(x, y, 1) / 255.0); p.pushdouble(t.getPixel(x, y, 2) / 255.0); p.pushdouble(t.getPixel(x, y, 3) / 255.0); break;
            case 2: p.foreground.r = t.getPixel(x, y, 0); p.foreground.g = t.getPixel(x, y, 1); p.foreground.b = t.getPixel(x, y, 2); p.foreground.a = t.getPixel(x, y, 3);
            case 3: p.background.r = t.getPixel(x, y, 0); p.background.g = t.getPixel(x, y, 1); p.background.b = t.getPixel(x, y, 2); p.background.a = t.getPixel(x, y, 3);
            case 128: t.setPixel(x, y, 3, p.popint()); t.setPixel(x, y, 2, p.popint()); t.setPixel(x, y, 1, p.popint()); t.setPixel(x, y, 0, p.popint()); break;
            case 129: t.setPixel(x, y, 3, int(p.popdouble() * 255)); t.setPixel(x, y, 2, int(p.popdouble() * 255)); t.setPixel(x, y, 1, int(p.popdouble() * 255)); t.setPixel(x, y, 0, int(p.popdouble() * 255)); break;
            case 130: t.setPixel(x, y, p.foreground); break;
            case 131: t.setPixel(x, y, p.background); break;
            default: p.invalid();
          }
        }
        else if(R == 243)
        {
          if(G == 0) { p.pushint(p.getTexture().getU()); p.pushint(p.getTexture().getV()); }
        }
        else if(R == 244) //texture alpha channel modifications
        {
          if(G == 0)
          {
            AlphaEffect ae(1, B, RGB_Black);
            createImageAlpha(&p.getTexture().getOpenGLBuffer()[0], p.getTexture().getU(), p.getTexture().getV(), ae);
          }
          else if(G == 1)
          {
            ColorRGB color;
            Texture& t = p.getTexture();
            for(int y = 0; y < t.getU(); y++)
            for(int x = 0; x < t.getU(); x++)
            {
              t.getPixel(color, x, y);
              if(color == p.foreground)
              {
                t.setPixel(x, y, p.background);
              }
            }
          }
        }
        else if(R == 255)
        {
          //"bright" NOP
        }
        else
        {
          p.invalid();
          //nothing
        }
      }
    }
    
    if(p.ip.dirx == 0 && p.ip.diry == 0)
    {
      redraw();
      SDL_Delay(5); //free CPU a bit when program is ended
    }
    
    p.moveIP();
  }
  
  std::cout << "counter: " << counter << "\n";
  
  SDL_Quit();
  return 0;
}



/*
suggesties voor nieuwe commandos:

-random nummer
-IP direction op andere dingen zetten
-er is memory leak ofzo wanneer screen meermaals achter elkaar wordt opgeroepen
-GL_CULL_FACE
-color conversions: rgb, hsv, hsl, ...
-commando om positie van MP relatief tov IP te zetten, voor "verplaatsbare code stukken die ook mee verplaatsende texture hebben"
-mouse coordinate change since last frame
*/
