/*
Copyright (c) 2005-2006 Lode Vandevenne
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
  * Neither the name of Lode Vandevenne nor the names of his contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "lpi_screen.h"

namespace lpi
{

void initBuiltInFontTextures(); //link time dependency to init the built in font textures
namespace gui { void initBuiltInGuiTextures(); } //link time dependency to init the built in gui textures

SDL_Surface *scr = 0; //the single SDL surface used
int w; //width of the screen
int h; //height of the screen

//these values are in OpenGL viewport coordinates, that is NOT the same as pixel coordinates, use setScissor to properly set these
std::vector<int> clipLeft;
std::vector<int> clipTop;
std::vector<int> clipRight;
std::vector<int> clipBottom;
bool fullscreenMode; //if true, it's fullscreen

int screenMode = -1;

/*
This function sets up an SDL window ready for OpenGL graphics.
You can choose the resolution, whether or not it's fullscreen, and a caption for the window.
*/
void screen(int width, int height, bool fullscreen, char *text)
{
  int colorDepth = 32;
  w = width;
  h = height;
  screenMode = -1;

  if(scr != 0) SDL_FreeSurface(scr);

  if(SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO) < 0)
  {
    printf("Unable to init SDL: %s\n", SDL_GetError());
    SDL_Quit();
    exit(1);
  }
  //atexit(SDL_Quit);
  if(fullscreen)
  {
    scr = SDL_SetVideoMode(width,height,colorDepth,SDL_SWSURFACE|SDL_FULLSCREEN|SDL_OPENGL);
    lock();
    fullscreenMode = 1;
  }
  else
  {
    scr = SDL_SetVideoMode(width,height,colorDepth,SDL_HWSURFACE|SDL_HWPALETTE|SDL_OPENGL);
    fullscreenMode = 0;
  }
  if(scr == 0)
  {
    printf("Unable to set video: %s\n", SDL_GetError());
    SDL_Quit();
    exit(1);
  }
  SDL_WM_SetCaption(text, NULL);
  
  initGL();
  
  SDL_EnableUNICODE(1); //for the text input things
  
  cls();
  
  //initialize the scissor area (the values at position std::vector.size() - 1 in the std::vectors must ALWAYS be set to the values below and may never be changed!)
  clipLeft.clear();
  clipLeft.push_back(0);
  clipTop.clear();
  clipTop.push_back(0);
  clipRight.clear();
  clipRight.push_back(w);
  clipBottom.clear();
  clipBottom.push_back(h);
  
  //plane.create(RGB_Black, w, h);
  
  initBuiltInFontTextures();
  //gui::initBuiltInGuiTextures();
  
}

double lastNear, lastFar;
void set2DScreen()
{
  if(screenMode == 0) return;
  //the official code for "Setting Your Raster Position to a Pixel Location" (i.e. set up an oldskool 2D screen)
  glViewport(0, 0, w, h);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(0, w, h, 0, -1, 1);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  screenMode = 0;
  
  enableTwoSided(); //important, without this, 2D stuff might not be drawn if only one side is enabled
  disableZBuffer();
}

//Initialize OpenGL: set up the camera and settings to emulate 2D graphics
void initGL()
{
  set2DScreen();
  
  //glShadeModel(GL_FLAT); //shading, don't do the GL_FLAT thing or gradient rectangles don't work anymore
  //glCullFace(GL_BACK); //culling
  //glFrontFace(GL_CCW);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);
  glDisable(GL_ALPHA_TEST);
  
  glEnable(GL_SCISSOR_TEST); //scissoring is used to, for example, not draw parts of textures that are scrolled away, and is always enabled (but by default the scissor area is as big as the screen)
  
}

void enableOneSided()
{
   glCullFace(GL_BACK);
   glEnable(GL_CULL_FACE);
}

void enableTwoSided()
{
  glDisable(GL_CULL_FACE);
}

bool filterTheTextures = false;

void enableTextureFiltering()
{
  filterTheTextures = true;
}

void disableTextureFiltering()
{
  filterTheTextures = false;
}

void toggleTextureFiltering()
{
  filterTheTextures = !filterTheTextures;
}

/*void bindTexture(Texture &t)
{
  glBindTexture(GL_TEXTURE_2D, t.texture[0]);
  if(filterTheTextures)
  {
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  }
  else
  {
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  }
}*/

void enableZBuffer()
{
  glEnable(GL_DEPTH_TEST);
}

void disableZBuffer()
{
  glDisable(GL_DEPTH_TEST);
}

//set new scissor area (limited drawing area on the screen)
void setScissor(int left, int top, int right, int bottom)
{
  if(right < left) right = left;
  if(bottom < top) bottom = top;
  
  //the values stored in the std::vectors are transformed to opengl viewport coordinates, AND right and bottom actually contain the size instead of coordinates (because OpenGL works that way)
  clipLeft.insert(clipLeft.begin(), left);
  clipTop.insert(clipTop.begin(), top);
  clipRight.insert(clipRight.begin(), right);
  clipBottom.insert(clipBottom.begin(), bottom);
  
  setOpenGLScissor();
}

void setSmallestScissor(int left, int top, int right, int bottom)
{
  int smallestLeft = left;
  int smallestTop = top;
  int smallestRight = right;
  int smallestBottom = bottom;
  
  
  if(clipLeft[0] > smallestLeft) smallestLeft = clipLeft[0]; //de meest rechtse van de linkerzijden
  if(clipTop[0] > smallestTop) smallestTop = clipTop[0]; //de laagste van de top zijden
  if(clipRight[0] < smallestRight) smallestRight = clipRight[0]; //de meest linkse van de rechtse zijden
  if(clipBottom[0] < smallestBottom) smallestBottom = clipBottom[0]; //de hoogste van de bodem zijden
  
  //if(smallestLeft < smallestRight) smallestLeft = 0, smallestRight = 1;
  //if(smallestTop < smallestBottom) smallestTop = 0, smallestBottom = 1;
  
  
  setScissor(smallestLeft, smallestTop, smallestRight, smallestBottom);
}

//uses the extern scissor area variables to set the scissoring area of OpenGL
void setOpenGLScissor()
{
  glScissor(clipLeft[0], h - clipBottom[0], clipRight[0] - clipLeft[0], clipBottom[0] - clipTop[0]);
}

//reset the scissor area back to the previous coordinates before your last setScissor call (works like a stack)
void resetScissor()
{
  if(clipLeft.size() > 1)
  {
    clipLeft.erase(clipLeft.begin());
    clipTop.erase(clipTop.begin());
    clipRight.erase(clipRight.begin());
    clipBottom.erase(clipBottom.begin());
  }
  
  setOpenGLScissor();
}

//Locks the screen
void lock()
{
  if(SDL_MUSTLOCK(scr))
  if(SDL_LockSurface(scr) < 0)
  return;
}

//Unlocks the screen
void unlock()
{
  if(SDL_MUSTLOCK(scr))
  SDL_UnlockSurface(scr);
}

//make the content that you drew to the backbuffer visible by swapping the buffers
void redraw()
{
  SDL_GL_SwapBuffers();
}

//clear the screen to black again
void cls(const ColorRGB& color)
{
  glClearColor(color.r / 255.0, color.g / 255.0, color.b / 255.0, 0);  //the clear color
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
}

//Draw an untextured, filled, rectangle on screen from (x1, y1) to (x2, y2)
void drawRectangle(int x1, int y1, int x2, int y2, const ColorRGB& color)
{
  x2++;
  y2++;
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);
  glDisable(GL_TEXTURE_2D);

  glColor4f(color.r / 255.0, color.g / 255.0, color.b / 255.0, color.a / 255.0);
  
  setOpenGLScissor(); //everything that draws something must always do this

  glBegin(GL_QUADS);
    glVertex3d(x2, y1, 1);
    glVertex3d(x1, y1, 1);
    glVertex3d(x1, y2, 1);
    glVertex3d(x2, y2, 1);
  glEnd();
}

//Draw a rectangle with 4 different corner colors on screen from (x1, y1) to (x2, y2)
void gradientRectangle(int x1, int y1, int x2, int y2, const ColorRGB& color1, const ColorRGB& color2, const ColorRGB& color3, const ColorRGB& color4)
{
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);
  glDisable(GL_TEXTURE_2D);
  
  setOpenGLScissor(); //everything that draws something must always do this
  
  glBegin(GL_QUADS);
    glColor4f(color2.r / 255.0, color2.g / 255.0, color2.b / 255.0, color2.a / 255.0);
    glVertex3d(x2, y1, 1);
    glColor4f(color1.r / 255.0, color1.g / 255.0, color1.b / 255.0, color1.a / 255.0);
    glVertex3d(x1, y1, 1);
    glColor4f(color3.r / 255.0, color3.g / 255.0, color3.b / 255.0, color3.a / 255.0);
    glVertex3d(x1, y2, 1);
    glColor4f(color4.r / 255.0, color4.g / 255.0, color4.b / 255.0, color4.a / 255.0);
    glVertex3d(x2, y2, 1);
  glEnd();
}

//Draw a line from (x1, y1) to (x2, y2) on the OpenGL screen
void drawLine(int x1, int y1, int x2, int y2, const ColorRGB& color, int clipx1, int clipy1, int clipx2, int clipy2)
{
  //clip if some point is outside the clipping area
  if(x1 < clipx1 || x1 >= clipx2 || x2 < clipx1 || x2 >= clipx2 || y1 < clipy1 || y1 >= clipy2 || y2 < clipy1 || y2 >= clipy2)
  {
    int x3, y3, x4, y4;
    if(!clipLine(x1, y1, x2, y2, x3, y3, x4, y4, clipx1, clipy1, clipx2, clipy2)) return;
    x1 = x3;
    y1 = y3;
    x2 = x4;
    y2 = y4;
  }
  
  //draw the line with OpenGL
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);
  glDisable(GL_TEXTURE_2D);

  glColor4ub(color.r, color.g, color.b, color.a);
  
  setOpenGLScissor(); //everything that draws something must always do this

     
  glBegin(GL_LINES);
    glVertex2d(x1, y1);
    glVertex2d(x2, y2);
  glEnd();
}

void pset(int x, int y, const ColorRGB& color)
{
  //draw the point with OpenGL
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);
  glDisable(GL_TEXTURE_2D);

  glColor4f(color.r / 255.0, color.g / 255.0, color.b / 255.0, color.a / 255.0);
  
  setOpenGLScissor(); //everything that draws something must always do this

  glBegin(GL_POINTS);
    glVertex2d(x + 0.375, y + 0.375);
  glEnd();
}

void drawPoint(int x, int y, const ColorRGB& color)
{
  pset(x, y, color);
}

//Draw a line from (x1, y1) to (x2, y2) on a buffer where the color is 4 unsigned chars (with bresenham)
void drawLine(unsigned char* buffer, int w, int h, int x1, int y1, int x2, int y2, const ColorRGB& color, int clipx1, int clipy1, int clipx2, int clipy2)
{
  //clip if some point is outside the clipping area
  if(x1 < clipx1 || x1 >= clipx2 || x2 < clipx1 || x2 >= clipx2 || y1 < clipy1 || y1 >= clipy2 || y2 < clipy1 || y2 >= clipy2)
  {
    int x3, y3, x4, y4;
    if(!clipLine(x1, y1, x2, y2, x3, y3, x4, y4, clipx1, clipy1, clipx2, clipy2)) return;
    x1 = x3;
    y1 = y3;
    x2 = x4;
    y2 = y4;
  }


  //draw the line with bresenham  
  int deltax = abs(x2 - x1);    // The difference between the x's
  int deltay = abs(y2 - y1);    // The difference between the y's
  int x = x1;           // Start x off at the first pixel
  int y = y1;           // Start y off at the first pixel
  int xinc1, xinc2, yinc1, yinc2, den, num, numadd, numpixels, curpixel;

  if (x2 >= x1)         // The x-values are increasing
  {
    xinc1 = 1;
    xinc2 = 1;
  }
  else              // The x-values are decreasing
  {
    xinc1 = -1;
    xinc2 = -1;
  }
  if (y2 >= y1)         // The y-values are increasing
  {
    yinc1 = 1;
    yinc2 = 1;
  }
  else              // The y-values are decreasing
  {
    yinc1 = -1;
    yinc2 = -1;
  }
  if (deltax >= deltay)     // There is at least one x-value for every y-value
  {
    xinc1 = 0;          // Don't change the x when numerator >= denominator
    yinc2 = 0;          // Don't change the y for every iteration
    den = deltax;
    num = deltax / 2;
    numadd = deltay;
    numpixels = deltax;     // There are more x-values than y-values
  }
  else              // There is at least one y-value for every x-value
  {
    xinc2 = 0;          // Don't change the x for every iteration
    yinc1 = 0;          // Don't change the y when numerator >= denominator
    den = deltay;
    num = deltay / 2;
    numadd = deltax;
    numpixels = deltay;     // There are more y-values than x-values
  }
  for (curpixel = 0; curpixel <= numpixels; curpixel++)
  {
    pset(buffer, w, h, x, y, color);  // Draw the current pixel
    num += numadd;        // Increase the numerator by the top of the fraction
    if (num >= den)       // Check if numerator >= denominator
    {
      num -= den;       // Calculate the new numerator value
      x += xinc1;       // Change the x as appropriate
      y += yinc1;       // Change the y as appropriate
    }
    x += xinc2;         // Change the x as appropriate
    y += yinc2;         // Change the y as appropriate
  }
}

//Set a pixel on a buffer where the color is 4 unsigned chars (w and h are the width and height of the buffer)
void pset(unsigned char* buffer, int w, int h, int x, int y, const ColorRGB& color)
{
  if(x < 0 || y < 0 || x >= w || y >= h) return;
  
  int bufferPos = 4 * w * y + 4 * x;
  buffer[bufferPos + 0] = color.r;
  buffer[bufferPos + 1] = color.g;
  buffer[bufferPos + 2] = color.b;
  buffer[bufferPos + 3] = color.a;
}

bool onScreen(int x, int y)
{
  return (x >= 0 && y >= 0 && x < w && y < h);
}

//Functions for clipping a 2D line to the screen, which is the rectangle (0,0)-(w,h)
//This is the Cohen-Sutherland Clipping Algorithm
//Each of 9 regions gets an outcode, based on if it's at the top, bottom, left or right of the screen
// 1001 1000 1010  9 8 10
// 0001 0000 0010  1 0 2
// 0101 0100 0110  5 4 6
//int findregion returns which of the 9 regions a point is in, void clipline does the actual clipping
int findRegion(int x, int y, int left, int top, int right, int bottom)
{
  int code=0;
  if(y >= bottom)
  code |= 1; //top
  else if( y < top)
  code |= 2; //bottom
  if(x >= right)
  code |= 4; //right
  else if ( x < left)
  code |= 8; //left
  return(code);
}
bool clipLine(int x1, int y1, int x2, int y2, int & x3, int & y3, int & x4, int & y4, int left, int top, int right, int bottom)
{
  int code1, code2, codeout;
  bool accept = 0, done = 0;
  code1 = findRegion(x1, y1, left, top, right, bottom); //the region outcodes for the endpoints
  code2 = findRegion(x2, y2, left, top, right, bottom);
  do  //In theory, this can never end up in an infinite loop, it'll always come in one of the trivial cases eventually
  {
    if(!(code1 | code2)) accept = done = 1;  //accept because both endpoints are in screen or on the border, trivial accept
    else if(code1 & code2) done = 1; //the line isn't visible on screen, trivial reject
    else  //if no trivial reject or accept, continue the loop
    {
      int x, y;
      codeout = code1 ? code1 : code2;
      if(codeout & 1) //top
      {
        x = x1 + (x2 - x1) * (bottom - y1) / (y2 - y1);
        y = bottom - 1;
      }
      else if(codeout & 2) //bottom
      {
        x = x1 + (x2 - x1) * (top - y1) / (y2 - y1);
        y = top;
      }
      else if(codeout & 4) //right
      {
        y = y1 + (y2 - y1) * (right - x1) / (x2 - x1);
        x = right - 1;
      }
      else //left
      {
        y = y1 + (y2 - y1) * (left - x1) / (x2 - x1);
        x = left;
      }
      if(codeout == code1) //first endpoint was clipped
      {
        x1 = x; y1 = y;
        code1 = findRegion(x1, y1, left, top, right, bottom);
      }
      else //second endpoint was clipped
      {
        x2 = x; y2 = y;
        code2 = findRegion(x2, y2, left, top, right, bottom);
      }
    }
  }
  while(done == 0);

  if(accept)
  {
    x3 = x1;
    x4 = x2;
    y3 = y1;
    y4 = y2;
    return 1;
  }
  else
  {
    x3 = x4 = y3 = y4 = 0;
    return 0;
  }
}

//Draw a gradient line from (x1, y1) to (x2, y2)
void gradientLine(int x1, int y1, int x2, int y2, const ColorRGB& color1, const ColorRGB& color2)
{
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);
  glDisable(GL_TEXTURE_2D);

  glBegin(GL_LINES);
    glColor4f(color1.r / 255.0, color1.g / 255.0, color1.b / 255.0, color1.a / 255.0);
    glVertex3d(x1, h - y1, 1);
    glColor4f(color2.r / 255.0, color2.g / 255.0, color2.b / 255.0, color2.a / 255.0);
    glVertex3d(x2, h - y2, 1);
  glEnd();
}

} //end of namespace lpi
