/*
Copyright (c) 2005-2006 Lode Vandevenne
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
  * Neither the name of Lode Vandevenne nor the names of his contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef math3d_included
#define math3d_included

#include "lpi_screen.h"

////////////////////////////////////////////////////////////////////////////////
//                  [ x ]                                                     //
// 2D Vector Class  [ y ]                                                     //
////////////////////////////////////////////////////////////////////////////////
class Vector2
{
  public:
         
  double x;
  double y;
  
  Vector2(double x, double y)
  {
    this->x = x;
    this->y = y;
  }
  Vector2()
  {
    x = 0.0;
    y = 0.0;
  }
};

Vector2 operator*(Vector2 v, double a);
Vector2 operator*(double a, Vector2 v);
Vector2 operator/(Vector2 v, double a);
Vector2 operator+(Vector2 v, Vector2 w);
Vector2 operator-(Vector2 v, Vector2 w);
Vector2 operator-(Vector2 v);


////////////////////////////////////////////////////////////////////////////////
//                  [ x ]                                                     //
// 3D Vector Class  [ y ]                                                     //
//                  [ z ]                                                     //
////////////////////////////////////////////////////////////////////////////////
class Vector3
{
  public:
         
  double x;
  double y;
  double z;
  
  Vector3(double x, double y, double z);
  Vector3();
  
  double length() const;
  double lengthsq() const;
  void normalize(); //normalizes the vector, but lets it remain the 0 vector if it's length is 0
  double distance(Vector3 v);
  double dot(Vector3 v);
  Vector3 cross(Vector3 v);
};

double length(const Vector3& v);
double lengthsq(const Vector3& v);
Vector3 normalize(Vector3 v);
double distance(Vector3 v, Vector3 w);
double distancesq(Vector3 v, Vector3 w);
double dot(Vector3 v, Vector3 w);
Vector3 cross(Vector3 v, Vector3 w);
Vector3 operator-(Vector3 v, Vector3 w);
Vector3 operator-(Vector3 v);
Vector3 operator+(Vector3 v, Vector3 w);
Vector3 operator*(Vector3 v, double a);
Vector3 operator*(double a, Vector3 v);
Vector3 operator/(Vector3 v, double a);
void operator+=(Vector3 &v, Vector3 w);
void operator-=(Vector3 &v, Vector3 w);
void operator*=(Vector3 &v, double a);
void operator/=(Vector3 &v, double a);
double vectorAngle(Vector3 v, Vector3 w);
double vectorAngleInDirection(Vector3 v, Vector3 w, Vector3 n);
Vector3 rotateAroundArbitrary(const Vector3& v, const Vector3& axis, double angle);
bool operator==(Vector3 a, Vector3 b);
bool operator!=(Vector3 a, Vector3 b);

#define Vector3_origin Vector3(0.0, 0.0, 0.0)
#define Vector3_0 Vector3(0.0, 0.0, 0.0)
#define Vector3_x Vector3(1.0, 0.0, 0.0)
#define Vector3_y Vector3(0.0, 1.0, 0.0)
#define Vector3_z Vector3(0.0, 0.0, 1.0)

////////////////////////////////////////////////////////////////////////////////
//               [ 0 3 6 ]                                                    //
// Matrix Class  [ 1 4 7 ]                                                    //
//               [ 2 5 8 ]                                                    //
////////////////////////////////////////////////////////////////////////////////
class Matrix3
{
  public:
    
  double a[9];
  
  Matrix3(double a0, double a1, double a2, double a3, double a4, double a5, double a6, double a7, double a8);
  Matrix3();
  
  void transpose();
  double determinant();
  void invert();
};

#define Matrix3_identity Matrix3(1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0)
#define Matrix3_unit Matrix3(1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0)
#define Matrix3_invZ Matrix3(1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, -1.0)

Matrix3 transpose(Matrix3 A);
double determinant(Matrix3 A);
Matrix3 inverse(Matrix3 A);
Matrix3 operator+(Matrix3 A, Matrix3 B);
Matrix3 operator-(Matrix3 A, Matrix3 B);
Matrix3 operator*(Matrix3 A, double a);
Matrix3 operator*(double a, Matrix3 A);
Matrix3 operator/(Matrix3 A, double a);
Vector3 operator*(Matrix3 A, Vector3 v);
//Vector3 operator*(Vector3 v, Matrix3 A);
Matrix3 operator*(Matrix3 A, Matrix3 B);

Matrix3 getRotationMatrix3(Vector3 axis, double angle);
Matrix3 rotateAroundArbitrary(Matrix3 m, Vector3 axis, double angle);

////////////////////////////////////////////////////////////////////////////////
//                        [ u.x v.x dir.x ]   [ pos.x ]                       //
// Transformation3 Class  [ u.y v.y dir.y ] + [ pos.y ]                       //
//                        [ u.z v.z dir.z ]   [ pos.z ]                       //
//                                                                            //
// This class can also be used as Camera                                      //
////////////////////////////////////////////////////////////////////////////////
class Transformation3
{

  public:
  double nearClip; //near clipping plane
  double farClip; //far clipping plane

  Transformation3();
  Transformation3(double posx, double posy, double posz,
       double ux,   double uy,   double uz,
       double vx,   double vy,   double vz,
       double dirx, double diry, double dirz,
       double nearClip = 0.0, double farClip = 0.0);
           
  //Get and Set the 4 important vectors 
  Vector3& getU(); //the "right" vector of the camera, x coordinate of screen
  Vector3& getV(); //the "up" vector of the camera, y coordinate of screen
  Vector3& getDir(); //the direction of the camera, vector that points into the screen (z direction)
  Vector3& getPos(); //the position of the camera in world coordinates
  void setDir(Vector3 newDir);
  void setU(Vector3 newY);
  void setV(Vector3 newV);  
  void setPos(Vector3 newPos);
  
  //move and rotate with vectors
  void move(const Vector3& offset);
  void rotate(const Vector3& axis, double angle);
  void rotate(const Vector3& a, const Vector3& b, double angle);
  void setLookDir(const Vector3& newDir);
  void lookAt(const Vector3& lookAtMe);
  void setGroundPlane(const Vector3& n);
  
  //get and set distance to a certain point
  double getDist(Vector3 point);
  void setDist(Vector3 point, double dist);
  
  //get and set zoom
  double getZoomU();
  double getZoomV();
  void setZoomU(double a);
  void setZoomV(double a); 
  void zoom(double a);
  void zoomU(double a);
  void zoomV(double a);
    
  //get and set FOV
  double getFOVU();
  double getFOVV();
  void setFOVU(double angle);
  void setFOVV(double angle);
  void setFOV(double angle)
  {
    setFOVU(angle);
    setFOVV(angle);
  }
  
  //get and set pitch, yaw and roll (these are NOT native parameters of the camera and should normally never be needed!)   
  double getYaw();
  double getPitch();
  double getRoll();
  void setYaw(double angle);
  void setPitch(double angle);
  void setRoll(double angle);
  void yawPlanet(double angle);
  void yawSpace(double angle);
  void pitch(double angle);
  void roll(double angle);
   
  //make camera orthogonal (reset skewing)
  void resetSkewU();
  void resetSkewV();
  
  //set screen ratio of the camera (ratio of length of u and v, e.g. 4:3, 16:9, 640:480, ...)
  double getRatioUV();
  double getRatioVU();
  void setRatioUV(double ratio);
  void setRatioVU(double ratio);
  
  //scale U, V and Dir without changing what you see
  double getScale();
  void setScale(double dirLength);
  void scale(double factor);
  
  //generate, get and use the camera matrix to transform points
  void generateMatrix();
  Matrix3 getMatrix();
  Matrix3 getInvMatrix();
  void setMatrix(Matrix3 matrix);   //sets the u, v and dir vector to given matrix (and generates the actual matrix too of course)
  Vector3 transform(Vector3 v);
  Vector3 transformPos(Vector3 v);
  bool projectOnScreen(Vector3 point, int & x, int & y, double & z);
  bool projectOnScreen(Vector3 point, int & x, int & y);
  bool transformOnScreen(Vector3 point, int & x, int & y, double & z);
  bool transformOnScreen(Vector3 point, int & x, int & y);
  bool camSpaceToScreen(Vector3 point, int & x, int & y);
  //same functions but for given w and h
  bool projectOnScreen(Vector3 point, int & x, int & y, double & z, int w, int h);
  bool projectOnScreen(Vector3 point, int & x, int & y, int w, int h);
  bool transformOnScreen(Vector3 point, int & x, int & y, double & z, int w, int h);
  bool transformOnScreen(Vector3 point, int & x, int & y, int w, int h);
  bool camSpaceToScreen(Vector3 point, int & x, int & y, int w, int h);
  /*//project on radar where the inner part is in front, outer part is outside
  void projectOnRadar(Vector3 point, int & x, int & y, int w, int h);*/
  
  private:
  //the camera plane, described by the vectors u and v, is described by "z = 0" in camera space
  Vector3 pos; //the location of the camera
  Vector3 u; //horizontal vector, horizontal side of computer screen
  Vector3 v; //vertical "up" vector, vertical side of computer screen
  Vector3 dir; //direction of the camera, direction of the projection
  Matrix3 transMatrix; //the camera matrix, is nothing more than the column vectors u, v and dir in one matrix
  Matrix3 invCamMatrix; //the inverse of the camera matrix
  bool matrixUpToDate;
  bool invMatrixUpToDate;
};


//Auxiliary functions

double radToDeg(double rad); //swap between radians and degrees
double degToRad(double deg);
Vector3 normalOfTriangle(Vector3 a, Vector3 b, Vector3 c); //calculate normal of a triangle
Vector3 projectOnPlane(Vector3 v, Vector3 n); //project vector v on the plane of which n is the normal
double rotationAroundAxis(Vector3 a, Vector3 b, Vector3 axis);
Vector3 getComponentInDirection(Vector3 v, Vector3 dir);
bool hasOppositeDirection(Vector3 v, Vector3 w);
bool sideOfPlaneGivenByNormal(Vector3 p, Vector3 n);
bool sideOfPlaneGivenByThreePoints(Vector3 p, Vector3 a, Vector3 b, Vector3 c);
bool sideOfPlaneGivenByTwoVectors(Vector3 p, Vector3 X, Vector3 Y);
double distance(Vector3 p, Vector3 a, Vector3 b);
double distance(Vector3 p, Vector3 a, Vector3 b, Vector3 c);
bool linesegmentThroughSphere(Vector3 a, Vector3 b, Vector3 p, double radius);
Vector3 linePlaneIntersection(Vector3 a, Vector3 b, Vector3 p, Vector3 q, Vector3 r);
void planeEquation(Vector3 a, Vector3 b, Vector3 c, double &u, double &v, double &w, double &t);
Vector3 getSomePerpendicularVector(Vector3 v);
Vector3 barycentric(Vector3 a, Vector3 b, Vector3 c, Vector3 p);
Vector3 rotateWithPlane(Vector3 p, Vector3 n);
Vector3 planeSphereTangent(Vector3 p, Vector3 o, double radius, Vector3 v);

////////////////////////////////////////////////////////////////////////////////

extern int screenMode; //0 = 2D screen, 1 = 3D screen, this is used by the set2DScreen and set3DScreen functions to only change when not changed to this mode already

void set3DScreen(double near, double far);
void setGLMatrix(Matrix3 m3, Vector3 pos);
void transformGLMatrix(Matrix3 m3, Vector3 pos);
void transformGLMatrixInvZ(Matrix3 m3, Vector3 pos);

////////////////////////////////////////////////////////////////////////////////

//4x4 math. It's a bit chaotically structured at the moment.

class Vector4
{
  public:
  double x;
  double y;
  double z;
  double w;
  
  void convertTo(Vector3& v)
  {
    v.x = x / w;
    v.y = y / w;
    v.z = z / w;
  }
  
  void convertFrom(const Vector3& v)
  {
    x = v.x;
    y = v.y;
    z = v.z;
    w = 1.0;
  }
};

class Matrix4
{
  public:
  
  double a[16];
  ////        ////
  // 0  4  8 12 //
  // 1  5  9 13 //
  // 2  6 10 14 //
  // 3  7 11 15 //
  ////        ////
  
  Matrix4() {}
  
  Matrix4(const Matrix4& m)
  {
    for(int i = 0; i < 16; i++) a[i] = m.a[i];
  }
  
  void operator=(const Matrix4& m)
  {
    for(int i = 0; i < 16; i++) a[i] = m.a[i];
  }  
  
  void transpose()
  {
    for(int i = 0; i < 16; i++)
    {
      int j = 4 * (i % 4) + (i / 4);
      if(i == j) continue;
      double temp = a[i];
      a[i] = a[j];
      a[j] = temp;
    }
  }
  
  void subMatrix(Matrix3& out, int i, int j )
  {
    int di, dj, si, sj;
    // loop through 3x3 submatrix
    for( di = 0; di < 3; di ++ )
    for( dj = 0; dj < 3; dj ++ )
    {
      // map 3x3 element (destination) to 4x4 element (source)
      si = di + ( ( di >= i ) ? 1 : 0 );
      sj = dj + ( ( dj >= j ) ? 1 : 0 );
      // copy element
      out.a[di * 3 + dj] = a[si * 4 + sj];
    }
  }
  
  double determinant()
  {
    double det, result = 0, i = 1;
    Matrix3 msub3;
    int n;
    for ( n = 0; n < 4; n++, i *= -1 )
    {
      subMatrix(msub3, 0, n);
      det = msub3.determinant();
      result += a[n] * det * i;
    }
    return(result);
  }
  
  void invert()
  {
    double mdet = determinant();
    Matrix3 mtemp;
    int sign;
    
    double result[16];

    for (int i = 0; i < 4; i++ )
    for (int j = 0; j < 4; j++ )
    {
      sign = 1 - ((i + j) % 2) * 2;
      subMatrix(mtemp, i, j);
      result[i+j*4] = (mtemp.determinant() * sign) / mdet;
    }
    
    for(int i = 0; i < 16; i++) a[i] = result[i];
  }
};

inline Matrix4 operator*(const Matrix4& A, const Matrix4& B)
{
  Matrix4 C;
  C.a[ 0] = A.a[0]*B.a[0]  + A.a[4]*B.a[1]  + A.a[8] *B.a[2]  + A.a[12]*B.a[3];
  C.a[ 1] = A.a[1]*B.a[0]  + A.a[5]*B.a[1]  + A.a[9] *B.a[2]  + A.a[13]*B.a[3];
  C.a[ 2] = A.a[2]*B.a[0]  + A.a[6]*B.a[1]  + A.a[10]*B.a[2]  + A.a[14]*B.a[3];
  C.a[ 3] = A.a[3]*B.a[0]  + A.a[7]*B.a[1]  + A.a[11]*B.a[2]  + A.a[15]*B.a[3];
  C.a[ 4] = A.a[0]*B.a[4]  + A.a[4]*B.a[5]  + A.a[8] *B.a[6]  + A.a[12]*B.a[7];
  C.a[ 5] = A.a[1]*B.a[4]  + A.a[5]*B.a[5]  + A.a[9] *B.a[6]  + A.a[13]*B.a[7];
  C.a[ 6] = A.a[2]*B.a[4]  + A.a[6]*B.a[5]  + A.a[10]*B.a[6]  + A.a[14]*B.a[7];
  C.a[ 7] = A.a[3]*B.a[4]  + A.a[7]*B.a[5]  + A.a[11]*B.a[6]  + A.a[15]*B.a[7];
  C.a[ 8] = A.a[0]*B.a[8]  + A.a[4]*B.a[9]  + A.a[8] *B.a[10] + A.a[12]*B.a[11];
  C.a[ 9] = A.a[1]*B.a[8]  + A.a[5]*B.a[9]  + A.a[9] *B.a[10] + A.a[13]*B.a[11];
  C.a[10] = A.a[2]*B.a[8]  + A.a[6]*B.a[9]  + A.a[10]*B.a[10] + A.a[14]*B.a[11];
  C.a[11] = A.a[3]*B.a[8]  + A.a[7]*B.a[9]  + A.a[11]*B.a[10] + A.a[15]*B.a[11];
  C.a[12] = A.a[0]*B.a[12] + A.a[4]*B.a[13] + A.a[8] *B.a[14] + A.a[12]*B.a[15];
  C.a[13] = A.a[1]*B.a[12] + A.a[5]*B.a[13] + A.a[9] *B.a[14] + A.a[13]*B.a[15];
  C.a[14] = A.a[2]*B.a[12] + A.a[6]*B.a[13] + A.a[10]*B.a[14] + A.a[14]*B.a[15];
  C.a[15] = A.a[3]*B.a[12] + A.a[7]*B.a[13] + A.a[11]*B.a[14] + A.a[15]*B.a[15];
  return C;
}

inline Vector3 operator*(const Matrix4& A, const Vector3& v)
{
  Vector3 result;
  double w;
  result.x = A.a[0] * v.x + A.a[4] * v.y + A.a[8]  * v.z + A.a[12];
  result.y = A.a[1] * v.x + A.a[5] * v.y + A.a[9]  * v.z + A.a[13];
  result.z = A.a[2] * v.x + A.a[6] * v.y + A.a[10] * v.z + A.a[14];
         w = A.a[3] * v.x + A.a[7] * v.y + A.a[11] * v.z + A.a[15];
  result /= w;
  return result;
}

inline Vector4 operator*(const Matrix4& A, const Vector4& v)
{
  Vector4 result;
  result.x = A.a[0] * v.x + A.a[4] * v.y + A.a[8]  * v.z + A.a[12];
  result.y = A.a[1] * v.x + A.a[5] * v.y + A.a[9]  * v.z + A.a[13];
  result.z = A.a[2] * v.x + A.a[6] * v.y + A.a[10] * v.z + A.a[14];
  result.w = A.a[3] * v.x + A.a[7] * v.y + A.a[11] * v.z + A.a[15];
  return result;
}

inline Matrix4 operator*(const Matrix4& A, double d)
{
  Matrix4 result;
  for(int i = 0; i < 16; i++) result.a[i] = d * A.a[i];
  return result;
}

inline Matrix4 operator*(double d, const Matrix4& A)
{
  Matrix4 result;
  for(int i = 0; i < 16; i++) result.a[i] = d * A.a[i];
  return result;
}

inline void getTranslationMatrix4(Matrix4& m, const Vector3& d)
{
  m.a[0] = 1; m.a[4] = 0; m.a[8] = 0; m.a[12] = d.x;
  m.a[1] = 0; m.a[5] = 1; m.a[9] = 0; m.a[13] = d.y;
  m.a[2] = 0; m.a[6] = 0; m.a[10] = 1; m.a[14] = d.z;
  m.a[3] = 0; m.a[7] = 0; m.a[11] = 0; m.a[15] = 1;
}

// inline void translate(Matrix4& m, const Vector3& d)
// {
//   //Matrix4 t; getTranslationMatrix4(t, d);
//   //m = t * m;
//   
//   Vector3 pos;
//   
//   pos.x = m.a[12];
//   pos.y = m.a[13];
//   pos.z = m.a[14];
//   pos /= m.a[15];
//   
//   Matrix3 m3;
//   m.subMatrix(m3, 3, 3);
//   m3.invert();
//   pos = m3 * pos;
//   
//   pos = pos + d;
//   
//   m3.invert();
//   pos = m3 * pos;
//   
//   m.a[12] = pos.x;
//   m.a[13] = pos.y;
//   m.a[14] = pos.z;
//   m.a[15] = 1;
//   
//   
// }

//returns a rotation matrix that rotates stuff by multiplying with this returned matrix, where you put this returned matrix on the left side of the multiplication.
inline void getRotationMatrix4(Matrix4& m, Vector3 axis, double angle)
{
  double c, s, t;

  axis.normalize();

  //calculate parameters of the rotation matrix
  c = cos(angle);
  s = sin(angle);
  t = 1 - c;
  
  m.a[0] = t * axis.x * axis.x +          c; m.a[4] = t * axis.x * axis.y + s * axis.z; m.a[8]  = t * axis.x * axis.z - s * axis.y; m.a[12] = 0.0;
  m.a[1] = t * axis.x * axis.y - s * axis.z; m.a[5] = t * axis.y * axis.y +          c; m.a[9]  = t * axis.y * axis.z + s * axis.x; m.a[13] = 0.0;
  m.a[2] = t * axis.x * axis.z + s * axis.y; m.a[6] = t * axis.y * axis.z - s * axis.x; m.a[10] = t * axis.z * axis.z +          c; m.a[14] = 0.0;
  m.a[3] = 0.0;                              m.a[7] = 0.0;                              m.a[11] = 0.0;                              m.a[15] = 1.0;
}

// inline void getPos(Vector3& pos, Matrix4 m)
// {
//   /*pos.x = -m.a[0]*m.a[12] - m.a[1]*m.a[13] - m.a[2]*m.a[14];
//   pos.y = -m.a[4]*m.a[12] - m.a[5]*m.a[13] - m.a[6]*m.a[14];
//   pos.z = -m.a[8]*m.a[12] - m.a[9]*m.a[13] - m.a[10]*m.a[14];*/
// 
//   /*m.invert();
//   pos.x = -m.a[12];
//   pos.y = -m.a[13];
//   pos.z = -m.a[14];
//   pos /= m.a[15];*/
//   
//   /*pos.x = m.a[12];
//   pos.y = m.a[13];
//   pos.z = m.a[14];
//   pos /= m.a[15];*/
//   
//   pos.x = m.a[12];
//   pos.y = m.a[13];
//   pos.z = m.a[14];
//   pos /= m.a[15];
//   
//   Matrix3 m3;
//   
//   m.subMatrix(m3, 3, 3);
//   
//   m3.invert();
//   
//   pos = m3 * pos;
// 
// }


inline void makeIdentity(Matrix4& m)
{
  m.a[0] = m.a[5] = m.a[10] = m.a[15] = 1;
  m.a[1] = m.a[2] = m.a[3] = 0;
  m.a[4] = m.a[6] = m.a[7] = 0;
  m.a[8] = m.a[9] = m.a[11] = 0;
  m.a[12] = m.a[13] = m.a[14] = 0;
}

#endif //math3d_included
