/*
Copyright (c) 2005-2006 Lode Vandevenne
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
  * Neither the name of Lode Vandevenne nor the names of his contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "lpi_input.h"
#include "lpi_general.h"


namespace lpi
{


Inkeys inkeys; //they keys that are pressed

void Inkeys::readKeys()
{
  keybuffer = SDL_GetKeyState(&size);
}

Inkeys::Inkeys()
{
  readKeys();
}

bool Inkeys::operator[](int index)
{
    return keybuffer[index];
}


SDL_Event event = {0}; //the events such as input, windows events, ...

/*
Gives value of pressed key to inkeys.
the variable inkeys can then be used anywhere to check for input.
Normally you have to use readkeys every time you want to use inkeys, but the
done() function also uses inkeys so it's not needed to use readkeys if you have 
a gameloop that uses done().

This function must be done at least once BEFORE ever trying to use inkeys, or you
will get an invalid result
*/
void readKeys()
{
  inkeys.readKeys();
}

//this returns true for a key only if it's pressed for the first time, pressing means "for the first time" while "down" would mean that they key is down at that moment
//index is there to be able to have different tests that don't screw up each other's values: give them a different index
bool keyReleased[256][16];
bool keyPressed(int key, int index) //use the SDL key code here, like SDLK_KP7 for keypad 7
{
  if(index < 0) index = 0;
  if(index > 15) index %= 16;

  if(inkeys[key] && keyReleased[key][index])
  {
    keyReleased[key][index] = 0;
    return 1;
  }
  if(!inkeys[key]) keyReleased[key][index] = 1;
  return 0;
}

bool keyDown(int key)
{
  readKeys();
  if(inkeys[key]) return 1;
  else return 0;
}

bool pressedEnter(int index)
{
  /*if((inkeys[SDLK_RETURN] || inkeys[SDLK_KP_ENTER]) && enterReleased)
  {
    enterReleased = 0;
    return 1;
  }
  if(!(inkeys[SDLK_RETURN] || inkeys[SDLK_KP_ENTER])) enterReleased = 1;
  return 0;*/
  if(keyPressed(SDLK_RETURN, index) || keyPressed(SDLK_KP_ENTER, index)) return 1;
  else return 0;
}


int mouseX;
int mouseY;
bool globalLMB;
bool globalRMB;
//mouse wheel state is checked in the done function in api.cpp because it can only checked with an sdl event
bool mouseWheelUp; //mouse wheel up
bool mouseWheelDown; //mouse wheel down

//get position and pressed buttons of the mouse, LMB=left mouse button, RMB=right mouse button
void checkMouse()
{
  Uint8 mouseState = SDL_GetMouseState(&mouseX, &mouseY);
  
  if(mouseState & 1) globalLMB = true; 
  else globalLMB = false;
  if(mouseState & 4) globalRMB = true; 
  else globalRMB = false;
}


int lastMouseX = -1, lastMouseY = -1;
/*
returns difference between current and last mouse x position, can be used only by one thing since calling it resets last position
this function doesn't call checkMouse so something else has already got to do it to make this work
*/
int mouseXDiff()
{
  int result = 0;
  if(lastMouseX >= 0) result = mouseX - lastMouseX;
  lastMouseX = mouseX;
  return result;
}

//idem for y direction
int mouseYDiff()
{
  int result = 0;
  if(lastMouseY >= 0) result = mouseY - lastMouseY;
  lastMouseY = mouseY;
  return result;
}

//same, but warps mouse back to old position every time
int mouseXDiffWarp()
{
  int result = 0;
  if(lastMouseX < 0) lastMouseX = mouseX;
  else
  {
    result = mouseX - lastMouseX;
    SDL_WarpMouse(lastMouseX, mouseY);
    mouseX = lastMouseX;
  }
  return result;
}

//idem for y direction
int mouseYDiffWarp()
{
  int result = 0;
  if(lastMouseY < 0) lastMouseY = mouseY;
  else
  {
    result = mouseY - lastMouseY;
    SDL_WarpMouse(mouseX, lastMouseY);
    mouseY = lastMouseY;
  }
  return result;
}

void resetMouseDiffFunctions()
{
  lastMouseX = lastMouseY = -1;
}

/*
This function returns which unicode key you're pressing, to type.
allowedChars disables or enables some characters such as numbers, special symbols, ...

If allowedChars is 0, all symbols are enabled
The following bits disable the following:
&1: disable punctuation symbols like ,(?.:
&2: disable numbers
&4: disable letters

Note that this function can also return things like when you pressed backspace, ...
*/

char previousChar[16];
float keyTime[16];
bool warmed_up[16];

Uint32 getTicks(); //defined in another .cpp file (link time dependency)

int unicodeKey(int allowedChars, int warmup, int rate, int index)
{
  if(index < 0) index = 0;
  if(index > 15) index %= 16;
  
  int asciiChar = 0;
  if ((event.key.keysym.unicode & 0xFF80) == 0)
  {
    if(event.type == SDL_KEYDOWN)
    {
      int inputChar = event.key.keysym.unicode & 0x7F;
      //handle numpad too, because it's not included in SDL's unicode thingie
      if(asciiChar == 0)
      {
        int keyPadTest = event.key.keysym.sym;
        switch(keyPadTest)
        {
          case SDLK_KP0: inputChar = '0'; break;
          case SDLK_KP1: inputChar = '1'; break;
          case SDLK_KP2: inputChar = '2'; break;
          case SDLK_KP3: inputChar = '3'; break;
          case SDLK_KP4: inputChar = '4'; break;
          case SDLK_KP5: inputChar = '5'; break;
          case SDLK_KP6: inputChar = '6'; break;
          case SDLK_KP7: inputChar = '7'; break;
          case SDLK_KP8: inputChar = '8'; break;
          case SDLK_KP9: inputChar = '9'; break;
          case SDLK_KP_PLUS: inputChar = '+'; break;
          case SDLK_KP_MINUS: inputChar = '-'; break;
          case SDLK_KP_MULTIPLY: inputChar = '*'; break;
          case SDLK_KP_DIVIDE: inputChar = '/'; break;
          case SDLK_KP_PERIOD: inputChar = '.'; break;
        }
      }

      //below is the system that prevents typing 100s of times the same char when holding down the key. It uses warmup and rate
      if(inputChar == previousChar[index])
      {
        asciiChar = 0;
        
        //if waited long enough, asciiChar can be set to inputChar anyway!
        if(getTicks() - keyTime[index] > warmup && !warmed_up[index])
        {
          keyTime[index] = getTicks();
          warmed_up[index] = 1;
        }
        else if(getTicks() - keyTime[index] > rate && warmed_up[index])
        {
          previousChar[index] = inputChar;
          asciiChar = inputChar;
          keyTime[index] = getTicks();
        }

      }
      else
      {
        previousChar[index] = inputChar;
        asciiChar = inputChar;
        keyTime[index] = getTicks();
        warmed_up[index] = 0;
      }
    }
    else
    previousChar[index] = 0; //so that you CAN press the same key twice in a row if you release it!
  }
  
  if(allowedChars & 1)
  {
    if(asciiChar < 0) asciiChar = 0; //signed extended ascii symbols
    if(asciiChar < 32 && asciiChar != 13 && asciiChar != 8) asciiChar = 0; //<32 ones, except enter and backspace
    if(asciiChar < 48 && asciiChar >= 32) asciiChar = 0; //before the numbers
    if(asciiChar > 57 && asciiChar < 65) asciiChar = 0; //between numbers and capitals
    if(asciiChar > 90 && asciiChar < 97) asciiChar = 0; //between capitals and small letters
    if(asciiChar > 122) asciiChar = 0; //after the small letters
  }
  if(allowedChars & 2)
  {
    if(asciiChar >= 47 && asciiChar <= 57) asciiChar = 0; //disable numbers too
  }
  if(allowedChars & 4)
  {
    if(asciiChar >= 65 && asciiChar <= 90) asciiChar = 0; //capitals
    if(asciiChar >= 97 && asciiChar <= 122) asciiChar = 0; //small letters
  }
  //disable unexpected special symbols except enter and backspace
  if(asciiChar < 32 && asciiChar != 13 && asciiChar != 8) asciiChar = 0; //<32 ones, except enter and backspace
  //some special ascii chars if you press left control with a keypad key
  readKeys();
  if(inkeys[SDLK_LCTRL] && !(allowedChars & 1))
  switch(asciiChar)
  {
    case '0': asciiChar = 128; break; //de franse c-dit
    case '1': asciiChar = 137; break; //e with trema:  zoals in Belgi
    case '2': asciiChar = 129; break; //u met 2 puntjes 
    case '3': asciiChar = 164; break; //n wiht ~, like the spanish seor 
    case '4': asciiChar = 168; break; //upside down question mark
    case '5': asciiChar = 173; break; //upside down exclamation mark
    case '6': asciiChar = 224; break; //alpha 
    case '7': asciiChar = 234; break; //large omega or ohm
    case '8': asciiChar = 227; break; //pi
    case '9': asciiChar = 230; break; //mu or micro
    case '/': asciiChar = 127; break; //pentagram
    case '*': asciiChar = 236; break; //infinity 
    case '-': asciiChar = 3; break; //heart
    case '+': asciiChar = 241; break; // +/-
    case '.': asciiChar = 1; break; //smiley
  }  
  return asciiChar;
}


//Waits until you press a key. First the key has to be loose, this means, if you put two sleep functions in a row, the second will only work after you first released the key.
void sleep()
{
  int done = 0;
  while(done == 0)
  {
    while(SDL_PollEvent(&event))
    {
      if(event.type == SDL_QUIT) end();
      if(event.type == SDL_KEYDOWN) done = 1;
    }
    SDL_Delay(5); //so it consumes less processing power
  }
}


/*
Returns 1 if you close the window or press the escape key
Also gets events and key input, so if you have done() in a loop it'll take
care of events as well.
*/
bool done()
{
  checkMouse(); //not needed for the done() function, but this way your loop automaticly checks the mouse every frame!

  int done = 0;
  if(!SDL_PollEvent(&event)) return 0;
  inkeys.readKeys();
  if(inkeys[SDLK_ESCAPE]) done = 1;
  if(event.type == SDL_QUIT) done = 1;
  
  mouseWheelUp = mouseWheelDown = false;
  if(event.type == SDL_MOUSEBUTTONDOWN)
  {
    if(event.button.button == SDL_BUTTON_WHEELUP) mouseWheelUp = true;
    if(event.button.button == SDL_BUTTON_WHEELDOWN) mouseWheelDown = true;
  }
  
  //SDL_Delay(5);
  
  return done;
}

} //namespace lpi

