/*
Copyright (c) 2005-2006 Lode Vandevenne
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
  * Neither the name of Lode Vandevenne nor the names of his contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef general_h
#define general_h 

#include <SDL/SDL.h>
#include <string>
#include <vector>
#include <cmath>
#include <iostream>
#include <algorithm>
#include <sstream>
#include <iomanip>

#include "lpi_color.h"

namespace lpi
{


////////////////////////////////////////////////////////////////////////////////
//NON GRAPHICAL FUNCTIONS///////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////


void end();

/*bool doubleClick(float time, bool button = 0);*/
Uint32 getTicks(); //returns the ticks in milliseconds
double getRandom();

////////////////////////////////////////////////////////////////////////////////
//3D DRAWING////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
void enableOneSided();
void enableTwoSided();
void enableZBuffer();
void disableZBuffer();
void enableTextureFiltering();
void disableTextureFiltering();
void toggleTextureFiltering();
extern bool filterTheTextures;

////////////////////////////////////////////////////////////////////////////////
//HANDY AUXILIARY FUNCTIONS/////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

void combSort(std::vector<int> &order, std::vector<double> &dist);
int wrapmod(int i, int n); //wraps i between 0 and n, using the modulo operator
double clamp(double a, double low, double high);
int clamp(int a, int low, int high);

//there are no proper versions of these functions, or at least I've had troubles with them, so I just define the same functions but with a different name to avoid conflicts (abs, max, min)


template<typename T>
const T template_abs(const T &a)
{
  return (a < 0) ? -a : a;
}

template<typename T>
const T template_max(const T &a, const T &b)
{
  return (a > b) ? a : b;
}

template<typename T>
const T template_min(const T &a, const T &b)
{
  return (a < b) ? a : b;
}



bool isPowerOfTwo(int n);
int floatMod(double f, int m);
int gcd(int a, int b);
float floatpow(float a, float b);

inline int intdown(double f) //intdown(0.5) becomes 0, intdown(-0.5) becomes -1, intdown(0.0) becomes something unknown, it's a border
{
  if(f > 0) return int(f);
  else return int(f) - 1;
}

inline Sint32 intdivdown(Sint32 a, Sint32 b) //divide so that 2/2 = 1, 1/2 = 0, -1/2 = -1, -2/2 = -1, ...
{
  //return a < 0 ? (a + b) / b - 1 : a / b;

  if(a < 0)
  {
    return -((-a - 1) / b + 1);
  }
  else return a / b;
}


///////////////////////////////////////////////////////////////////////////////

class GameTime
{
  public:
  Sint32 oldTime; //this is in milliseconds
  Sint32 newTime;
  
  //for smooth fps value
  #define NUMLASTTIMES 64
  std::vector<Uint32> lastTimes;
  
  GameTime()
  {
    init();
  }
  
  void init()
  {
    oldTime = getTicks();
    newTime = getTicks();
    lastTimes.clear();
    lastTimes.push_back(newTime);
    
    prepared = false;
  }
  
  Sint32 diff() //return difference between two last frames
  {
    return newTime - oldTime;
  }
  
  double diff_s() //return difference between two last frames, in seconds as floating point
  {
    return diff() / 1000.0;
  }
  
  Sint32 currentDiff() //return difference between last frame and current time
  {
    return getTicks() - newTime;
  }
  
  double fps() //returns fps from last NUMLASTTIMES frames instead of only the last one to have smoother values
  {
    
    double timeDiff = (lastTimes[lastTimes.size() - 1] - lastTimes[0]) / 1000.0; //time difference in seconds
    
    return 1.0 / (timeDiff / lastTimes.size());
    
  }
  
  void update()
  {
    oldTime = newTime;
    newTime = getTicks();
    
    //for the fps stuff
    if(lastTimes.size() >= NUMLASTTIMES || !prepared) lastTimes.erase(lastTimes.begin());
    lastTimes.push_back(newTime);
    
    //avoid weird initialization problems with the prepared system
    if(!prepared) prepared = true;
  }
  
  bool prepared;
};

void encodeBase64(const std::vector<unsigned char>& in, std::string& out);
void decodeBase64(const std::string& in, std::vector<unsigned char>& out);

} //namespace lpi

#endif //general_h
