/*
Copyright (c) 2005-2006 Lode Vandevenne
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
  * Neither the name of Lode Vandevenne nor the names of his contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef color_h
#define color_h

#include <cmath>
#include <algorithm> //min and max

namespace lpi
{
namespace color
{

//This does all the conversions from any color model to any color model

template<typename ColorTypeFrom, typename ColorTypeTo>
ColorTypeTo convert(const ColorTypeFrom& in)
{
  ColorTypeTo out;
  out.fromBase(in.toBase());
  return out;
}

template<typename ColorTypeFrom, typename ColorTypeTo>
void convert(const ColorTypeFrom& in, ColorTypeTo& out)
{
  out = convert<ColorTypeFrom, ColorTypeTo>(in);
}

//Base color classes used as intermediate result for conversions. (RGB)

template<typename T>
struct RGB255;

template<typename T>
struct RGB1;

template<typename T>
struct RGB255 //base color for integer colors
{
  T r;
  T g;
  T b;
  
  RGB255() {}
  
  RGB255(const T& r, const T& g, const T& b)
  {
    this->r = r;
    this->g = g;
    this->b = b;
  }

  RGB255<T> toBase() const { return *this; }
  void fromBase(const RGB255<T>& in) { *this = in; }

  //conversion to other types
  template<typename U>
  operator RGB255<U>()
  {
    RGB255<U> out;
    out.r = U(r);
    out.g = U(g);
    out.b = U(b);
    return out;
  }

  template<typename U>
  operator RGB1<U>()
  {
    RGB1<U> out;
    out.r = U(r / 255.0);
    out.g = U(g / 255.0);
    out.b = U(b / 255.0);
    return out;
  }
};

template<typename T>
void operator*=(RGB255<T>& a, const RGB255<T>& b)
{
  //the +1 is there so that if both are 255, after dividing through 256, the result is 255 and not 254, that's faster than division through 255
  a.r = (a.r * (b.r + 1)) / 256;
  a.g = (a.g * (b.g + 1)) / 256;
  a.b = (a.b * (b.b + 1)) / 256;
}

template<typename T>
bool operator==(const RGB255<T>& a, const RGB255<T>& b)
{
  return a.r == b.r && a.g == b.g && a.b == b.b;
}

template<typename T>
RGB255<T> operator*(const RGB255<T>& a, const RGB255<T>& b)
{
  RGB255<T> result = a;
  operator*=(result, b);
  return result;
}

template<typename T, typename Divider>
void operator/=(RGB255<T>& a, Divider d)
{
  a.r = (a.r / d);
  a.g = (a.g / d);
  a.b = (a.b / d);
}

template<typename T, typename Divider>
RGB255<T> operator/(const RGB255<T>& a, Divider d)
{
  RGB255<T> result = a;
  operator/=(result, d);
  return result;
}

template<typename T, typename Divider>
RGB255<T> operator*=(RGB255<T>& a, Divider d)
{
  a.r = (a.r * d);
  a.g = (a.g * d);
  a.b = (a.b * d);
}

template<typename T, typename Divider>
RGB255<T> operator*(const RGB255<T>& a, Divider d)
{
  RGB255<T> result = a;
  operator*=(result, d);
  return result;
}


template<typename T>
struct RGB1 //base color for colors using channels with range 0.0-1.0
{
  T r;
  T g;
  T b;

  RGB255<T> toBase() const { return *this; }
  void fromBase(const RGB255<T>& in) { *this = in; }

  //conversion to other types
  template<typename U>
  operator RGB1<U>()
  {
    RGB1<U> out;
    out.r = U(r);
    out.g = U(g);
    out.b = U(b);
    return out;
  }

  template<typename U>
  operator RGB255<U>()
  {
    RGB255<U> out;
    out.r = U(r * 255.0);
    out.g = U(g * 255.0);
    out.b = U(b * 255.0);
    return out;
  }
};

template<typename T>
void operator*=(RGB1<T>& a, const RGB1<T>& b)
{
  a.r = a.r * b.r;
  a.g = a.g * b.g;
  a.b = a.b * b.b;
}

template<typename T>
RGB1<T> operator*(const RGB1<T>& a, const RGB1<T>& b)
{
  RGB1<T> result = a;
  operator*=(result, b);
  return result;
}

//T must be integer for HSV255, due to the % operator used
template<typename T, typename BaseColor = RGB255<T> >
struct HSV255
{
  T h;
  T s;
  T v;
  
  HSV255() {}
  
  HSV255(const T& h, const T& s, const T& v)
  {
    this->h = h;
    this->s = s;
    this->v = v;
  }

  BaseColor toBase() const
  {
    BaseColor out;
    //If saturation is 0, the color is a shade of gray
    if(s == 0) out.r = out.g = out.b = v;

    //If saturation > 0, more complex calculations are needed
    else
    {
      T f, p, q, t, i;
      i = (h * 6) / 256;
      f = (h * 6) % 256;  //the fractional part of h
      p = (v * (255 - s)) / 256;
      q = (v * (255 - (s * f) / 256)) / 256;
      t = (v * (255 - ((s * (255 - f))) / 256)) / 256;
      switch(i)
      {
        case 0: out.r = v; out.g = t; out.b = p; break;
        case 1: out.r = q; out.g = v; out.b = p; break;
        case 2: out.r = p; out.g = v; out.b = t; break;
        case 3: out.r = p; out.g = q; out.b = v; break;
        case 4: out.r = t; out.g = p; out.b = v; break;
        case 5: out.r = v; out.g = p; out.b = q; break;
      }
    }
    return out;
  }

  void fromBase(const BaseColor& in)
  {
    T maxColor = std::max(in.r, std::max(in.g, in.b));
    T minColor = std::min(in.r, std::min(in.g, in.b));
    v = maxColor;
    if(maxColor == 0)//avoid division by zero when the color is black
    {
      s = 0;
    }
    else
    {
      s = (256 * (maxColor - minColor)) / maxColor;
    }

    if(s == 0)
    {
      h = 0; //it doesn't matter what value it has
    }
    else
    {
      if     (in.r == maxColor) h =        (256 * (in.g - in.b)) / (maxColor - minColor);
      else if(in.g == maxColor) h =  512 + (256 * (in.b - in.r)) / (maxColor - minColor);
      else                      h = 1024 + (256 * (in.r - in.g)) / (maxColor - minColor);
      h /= 6;
      if (h < 0) h++;
    }
  }
};

template<typename T, typename BaseColor = RGB1<T> >
struct HSV1
{
  T h;
  T s;
  T v;
  
  BaseColor toBase() const
  {
    BaseColor out;
    //If saturation is 0, the color is a shade of gray
    if(s == 0.0) out.r = out.g = out.b = v;

    //If saturation > 0, more complex calculations are needed
    else
    {
      T f, p, q, t;
      int i;
      i = int(std::floor(h * 6.0)); //choose 1 of 6 zones on hue circle
      f = h * 6.0 - i;  //the fractional part of h * 6
      p = v * (1.0 - s);
      q = v * (1.0 - (s * f));
      t = v * (1.0 - (s * (1.0 - f)));
      switch(i)
      {
        case 0: out.r = v; out.g = t; out.b = p; break;
        case 1: out.r = q; out.g = v; out.b = p; break;
        case 2: out.r = p; out.g = v; out.b = t; break;
        case 3: out.r = p; out.g = q; out.b = v; break;
        case 4: out.r = t; out.g = p; out.b = v; break;
        case 5: out.r = v; out.g = p; out.b = q; break;
      }
    }
    return out;
  }

  void fromBase(const BaseColor& in)
  {
    T maxColor = std::max(in.r, std::max(in.g, in.b));
    T minColor = std::min(in.r, std::min(in.g, in.b));
    v = maxColor;
    if(maxColor == 0.0)//avoid division by zero when the color is black
    {
      s = 0.0;
    }
    else
    {
      s = (maxColor - minColor) / maxColor;
    }

    if(s == 0.0)
    {
      h = 0; //it doesn't matter what value it has
    }
    else
    {
      if     (in.r == maxColor) h =       (in.g - in.b) / (maxColor - minColor);
      else if(in.g == maxColor) h = 2.0 + (in.b - in.r) / (maxColor - minColor);
      else                      h = 4.0 + (in.r - in.g) / (maxColor - minColor);
      h /= 6.0; //to bring it to a number between 0 and 1
      if (h < 0.0) h++;
    }
  }
};

template<typename T, typename BaseColor = RGB255<T> >
struct HSL255
{
  T h;
  T s;
  T l;
  
  HSL255() {}
  
  HSL255(T h, T s, T l)
  {
    this->h = h;
    this->s = s;
    this->l = l;
  }

  BaseColor toBase() const
  {
    BaseColor out;
    T temp1, temp2, tempr, tempg, tempb;

    if(s == 0) out.r = out.g = out.b = l; //If saturation is 0, the color is a shade of gray
    else
    {
      //Set the temporary values
      if(l < 128) temp2 = (l * (256 + s))           / 256;
      else        temp2 = (l +         s) - (l * s) / 256;
      temp1 = 2 * l - temp2;
      tempr = h + 256 / 3;
      if(tempr > 255) tempr -= 256;
      tempg = h;
      tempb = h - 256 / 3;
      if(tempb < 0) tempb += 256;

      //Red
      if     (tempr < 256 / 6) out.r = temp1 + ((temp2 - temp1) * 6 * tempr) / 256;
      else if(tempr < 128    ) out.r = temp2;
      else if(tempr < 512 / 3) out.r = temp1 + ((temp2 - temp1) * ((512 / 3) - tempr) * 6) / 256;
      else out.r = temp1;

      //Green
      if     (tempg < 256 / 6) out.g = temp1 + ((temp2 - temp1) * 6 * tempg) / 256;
      else if(tempg < 128    ) out.g = temp2;
      else if(tempg < 512 / 3) out.g = temp1 + ((temp2 - temp1) * ((512 / 3) - tempg) * 6) / 256;
      else out.g = temp1;

      //Blue
      if     (tempb < 256 / 6) out.b = temp1 + ((temp2 - temp1) * 6 * tempb) / 256;
      else if(tempb < 128    ) out.b = temp2;
      else if(tempb < 512 / 3) out.b = temp1 + ((temp2 - temp1) * ((512 / 3) - tempb) * 6) / 256;
      else out.b = temp1;
    }
    return out;
  }

  void fromBase(const BaseColor& in)
  {
    T maxColor = std::max(in.r, std::max(in.g, in.b));
    T minColor = std::min(in.r, std::min(in.g, in.b));

    if(minColor == maxColor) //r == g == b, so it's a shade of gray
    {
      h = 0; //it doesn't matter what value it has
      s = 0;
      l = in.r; //doesn't matter if you pick r, g, or b
    }
    else
    {
      l = (minColor + maxColor) / 2;

      if(l < 128) s = (256 * (maxColor - minColor)) / (maxColor + minColor);
      else        s = (256 * (maxColor - minColor)) / (512 - maxColor - minColor);

      if     (in.r == maxColor) h =        (256 * (in.g - in.b)) / (maxColor - minColor);
      else if(in.g == maxColor) h =  512 + (256 * (in.b - in.r)) / (maxColor - minColor);
      else                      h = 1024 + (256 * (in.r - in.g)) / (maxColor - minColor);

      h /= 6; //to bring it to a number between 0 and 1
      if(h < 0) h++;
    }
  }
};

template<typename T, typename BaseColor = RGB1<T> >
struct HSL1
{
  T h;
  T s;
  T l;

  BaseColor toBase() const
  {
    BaseColor out;
    T temp1, temp2, tempr, tempg, tempb;

    //If saturation is 0, the color is a shade of gray
    if(s == 0.0) out.r = out.g = out.b = l;

    //If saturation > 0, more complex calculations are needed
    else
    {
      //Set the temporary values
      if(l < 0.5) temp2 = l * (1.0 + s);
      else temp2 = (l + s) - (l * s);
      temp1 = 2.0 * l - temp2;
      tempr = h + 1.0 / 3.0;
      if(tempr > 1.0) tempr--;
      tempg = h;
      tempb = h - 1.0 / 3.0;
      if(tempb < 0.0) tempb++;

      //Red
      if(tempr < 1.0 / 6.0) out.r = temp1 + (temp2 - temp1) * 6.0 * tempr;
      else if(tempr < 0.5) out.r = temp2;
      else if(tempr < 2.0 / 3.0) out.r = temp1 + (temp2 - temp1) * ((2.0 / 3.0) - tempr) * 6.0;
      else out.r = temp1;

      //Green
      if(tempg < 1.0 / 6.0) out.g = temp1 + (temp2 - temp1) * 6.0 * tempg;
      else if(tempg < 0.5) out.g = temp2;
      else if(tempg < 2.0 / 3.0) out.g = temp1 + (temp2 - temp1) * ((2.0 / 3.0) - tempg) * 6.0;
      else out.g = temp1;

      //Blue
      if(tempb < 1.0 / 6.0) out.b = temp1 + (temp2 - temp1) * 6.0 * tempb;
      else if(tempb < 0.5) out.b = temp2;
      else if(tempb < 2.0 / 3.0) out.b = temp1 + (temp2 - temp1) * ((2.0 / 3.0) - tempb) * 6.0;
      else out.b = temp1;
    }
    return out;
  }

  void fromBase(const BaseColor& in)
  {
    T maxColor = std::max(in.r, std::max(in.g, in.b));
    T minColor = std::min(in.r, std::min(in.g, in.b));

    if(minColor == maxColor) //r == g == b, so it's a shade of gray
    {
      h = 0.0; //it doesn't matter what value it has
      s = 0.0;
      l = in.r; //doesn't matter if you pick r, g, or b
    }
    else
    {
      l = (minColor + maxColor) / 2.0;

      if(l < 0.5) s = (maxColor - minColor) / (maxColor + minColor);
      else s = (maxColor - minColor) / (2.0 - maxColor - minColor);

      if     (in.r == maxColor) h =       (in.g - in.b) / (maxColor - minColor);
      else if(in.g == maxColor) h = 2.0 + (in.b - in.r) / (maxColor - minColor);
      else                      h = 4.0 + (in.r - in.g) / (maxColor - minColor);

      h /= 6.0; //to bring it to a number between 0 and 1
      if(h < 0.0) h++;
    }
  }
};

template<typename T, typename BaseColor>
struct YUV
{
  T Y;
  T U;
  T V;

  BaseColor toBase() const
  {
    BaseColor out;
    out.r = T(Y - 0.000039457070707 * U + 1.139827967171717 * V);
    out.g = T(Y - 0.394610164141414 * U - 0.580500315656566 * V);
    out.b = T(Y + 2.031999684343434 * U - 0.000481376262626 * V);
    return out;
  }

  void fromBase(const BaseColor& in)
  {
    Y = T( 0.299 * in.r  + 0.587 * in.g + 0.114 * in.b);
    U = T(-0.147 * in.r  - 0.289 * in.g + 0.436 * in.b);
    V = T( 0.615 * in.r  - 0.515 * in.g - 0.100 * in.b);
  }
};

template<typename T, typename BaseColor>
struct YIQ
{
  T Y;
  T I;
  T Q;

  BaseColor toBase() const
  {
    BaseColor out;
    out.r = T(Y + 0.956295719758948 * I + 0.621024416465261 * Q);
    out.g = T(Y - 0.272122099318510 * I - 0.647380596825695 * Q);
    out.b = T(Y - 1.106989016736491 * I + 1.704614998364648 * Q);
    return out;
  }

  void fromBase(const BaseColor& in)
  {
    Y = T(0.299    * in.r + 0.587    * in.g + 0.114    * in.b);
    I = T(0.595716 * in.r - 0.274453 * in.g - 0.321263 * in.b);
    Q = T(0.211456 * in.r - 0.522591 * in.g + 0.311135 * in.b);
  }
};


template<typename T, typename BaseColor>
struct JPEG_YCbCr
{
  T Y;
  T Cb;
  T Cr;

  BaseColor toBase() const
  {
    BaseColor out;
    out.r = T(Y                        + 1.402   * (Cr - 128));
    out.g = T(Y - 0.34414 * (Cb - 128) - 0.71414 * (Cr - 128));
    out.b = T(Y + 1.772   * (Cb - 128)                       );
    return out;
  }

  void fromBase(const BaseColor& in)
  {
    Y  = T(    + 0.299    * in.r + 0.587    * in.g + 0.114    * in.b);
    Cb = T(128 - 0.168736 * in.r - 0.331264 * in.g + 0.5      * in.b);
    Cr = T(128 + 0.5      * in.r - 0.418688 * in.g - 0.081312 * in.b);
  }
};

template<typename T, typename BaseColor>
struct YPbPr
{
  T Y;
  T Pb;
  T Pr;

  BaseColor toBase() const
  {
    BaseColor out;
    out.r = T(Y                + 1.402   * Pr);
    out.g = T(Y - 0.34414 * Pb - 0.71414 * Pr);
    out.b = T(Y + 1.772   * Pb               );
    return out;
  }

  void fromBase(const BaseColor& in)
  {
    Y  = T(+ 0.299    * in.r + 0.587    * in.g + 0.114    * in.b);
    Pb = T(- 0.168736 * in.r - 0.331264 * in.g + 0.5      * in.b);
    Pr = T(+ 0.5      * in.r - 0.418688 * in.g - 0.081312 * in.b);
  }
};

template<typename T, typename BaseColor = RGB255<T> >
struct CMY255
{
  T C;
  T M;
  T Y;

  BaseColor toBase() const
  {
    BaseColor out;
    out.r = T(255 - C);
    out.g = T(255 - M);
    out.b = T(255 - Y);
    return out;
  }

  void fromBase(const BaseColor& in)
  {
    C = T(255 - in.r);
    M = T(255 - in.g);
    Y = T(255 - in.b);
  }
};

template<typename T, typename BaseColor = RGB1<T> >
struct CMY1
{
  T C;
  T M;
  T Y;

  BaseColor toBase() const
  {
    BaseColor out;
    out.r = T(1.0 - C);
    out.g = T(1.0 - M);
    out.b = T(1.0 - Y);
    return out;
  }

  void fromBase(const BaseColor& in)
  {
    C = T(1.0 - in.r);
    M = T(1.0 - in.g);
    Y = T(1.0 - in.b);
  }
};


template<typename T, typename BaseColor = RGB255<T> >
struct INT255 //integer color: r, g and b into a single integer (that must be at least 32 bit)
{
  T i; //has to be at least 32 bits

  BaseColor toBase() const
  {
    BaseColor out;
    out.r = (i / 65536) % 256;
    out.g = (i /   256) % 256;
    out.b = (i /     1) % 256;
    return out;
  }

  void fromBase(const BaseColor& in)
  {
    i = 65536 * in.r + 256 * in.g + in.b;
  }
};

////////////////////////////////////////////////////////////////////////////////

template<typename ColorType, typename AlphaType>
struct alpha255 : public ColorType
{
  AlphaType a;

  alpha255()
  {
  }

  alpha255(const ColorType& in)
  {
    this->ColorType::operator=(in);
  }
  
  alpha255(const AlphaType& C1, const AlphaType& C2, const AlphaType& C3, const AlphaType& a) : ColorType(C1, C2, C3)
  {
    this->a = a;
  }
  
  alpha255(const AlphaType& C1, const AlphaType& C2, const AlphaType& C3) : ColorType(C1, C2, C3)
  {
    this->a = 255;
  }

  operator ColorType&()
  {
    return *this;
  }

  void operator=(const ColorType& in)
  {
    this->ColorType::operator=(in);
  }
};

template<typename ColorType, typename AlphaType>
bool operator==(const alpha255<ColorType, AlphaType>& a, const alpha255<ColorType, AlphaType>& b)
{
  return a.a == b.a && static_cast<ColorType>(a) == static_cast<ColorType>(b);
}


} //end of namespace color 


typedef color::alpha255<color::RGB255<int>, int> ColorRGB;
typedef color::alpha255<color::HSV255<int>, int> ColorHSV;

static const ColorRGB RGB_Black        (  0,   0,   0, 255);
static const ColorRGB RGB_Red          (255,   0,   0, 255);
static const ColorRGB RGB_Green        (  0, 255,   0, 255);
static const ColorRGB RGB_Blue         (  0,   0, 255, 255);
static const ColorRGB RGB_Cyan         (  0, 255, 255, 255);
static const ColorRGB RGB_Magenta      (255,   0, 255, 255);
static const ColorRGB RGB_Yellow       (255, 255,   0, 255);
static const ColorRGB RGB_White        (255, 255, 255, 255);
static const ColorRGB RGB_Gray         (128, 128, 128, 255);
static const ColorRGB RGB_Grey         (192, 192, 192, 255);
static const ColorRGB RGB_Darkred      (128,   0,   0, 255);
static const ColorRGB RGB_Darkgreen    (  0, 128,   0, 255);
static const ColorRGB RGB_Darkblue     (  0,   0, 128, 255);
static const ColorRGB RGB_Darkcyan     (  0, 128, 128, 255);
static const ColorRGB RGB_Darkmagenta  (128,   0, 128, 255);
static const ColorRGB RGB_Darkyellow   (128, 128,   0, 255);
static const ColorRGB RGB_Darkgray     ( 64,  64,  64, 255);
static const ColorRGB RGB_Darkgrey     ( 96,  96,  96, 255);
static const ColorRGB RGB_Lightred     (255, 128, 128, 255);
static const ColorRGB RGB_Lightgreen   (128, 255, 128, 255);
static const ColorRGB RGB_Lightblue    (128, 128, 255, 255);
static const ColorRGB RGB_Lightcyan    (128, 255, 255, 255);
static const ColorRGB RGB_Lightmagenta (255, 128, 255, 255);
static const ColorRGB RGB_Lightyellow  (255, 255, 128, 255);
static const ColorRGB RGB_Lightgray    (224, 224, 224, 255);
static const ColorRGB RGB_Lightgrey    (240, 240, 240, 255);
static const ColorRGB RGB_Brightred    (255, 192, 192, 255);
static const ColorRGB RGB_Brightgreen  (192, 255, 192, 255);
static const ColorRGB RGB_Brightblue   (192, 192, 255, 255);
static const ColorRGB RGB_Brightcyan   (192, 255, 255, 255);
static const ColorRGB RGB_Brightmagenta(255, 192, 255, 255);
static const ColorRGB RGB_Brightyellow (255, 255, 192, 255);
static const ColorRGB RGB_Whitered     (255, 224, 224, 255);
static const ColorRGB RGB_Whitegreen   (224, 255, 224, 255);
static const ColorRGB RGB_Whiteblue    (224, 224, 255, 255);
static const ColorRGB RGB_Whitecyan    (224, 255, 255, 255);
static const ColorRGB RGB_Whitemagenta (255, 224, 255, 255);
static const ColorRGB RGB_Whiteyellow  (255, 255, 224, 255);
static const ColorRGB RGB_Orange       (255, 165,   0, 255);

#define RGBA_Black(a)        ColorRGB(  0,   0,   0, a)
#define RGBA_Red(a)          ColorRGB(255,   0,   0, a)
#define RGBA_Green(a)        ColorRGB(  0, 255,   0, a)
#define RGBA_Blue(a)         ColorRGB(  0,   0, 255, a)
#define RGBA_Cyan(a)         ColorRGB(  0, 255, 255, a)
#define RGBA_Magenta(a)      ColorRGB(255,   0, 255, a)
#define RGBA_Yellow(a)       ColorRGB(255, 255,   0, a)
#define RGBA_White(a)        ColorRGB(255, 255, 255, a)
#define RGBA_Gray(a)         ColorRGB(128, 128, 128, a)
#define RGBA_Grey(a)         ColorRGB(192, 192, 192, a)
#define RGBA_Darkred(a)      ColorRGB(128,   0,   0, a)
#define RGBA_Darkgreen(a)    ColorRGB(  0, 128,   0, a)
#define RGBA_Darkblue(a)     ColorRGB(  0,   0, 128, a)
#define RGBA_Darkcyan(a)     ColorRGB(  0, 128, 128, a)
#define RGBA_Darkmagenta(a)  ColorRGB(128,   0, 128, a)
#define RGBA_Darkyellow(a)   ColorRGB(128, 128,   0, a)
#define RGBA_Darkgray(a)     ColorRGB( 64,  64,  64, a)
#define RGBA_Darkgrey(a)     ColorRGB( 96,  96,  96, a)
#define RGBA_Lightred(a)     ColorRGB(255, 128, 128, a)
#define RGBA_Lightgreen(a)   ColorRGB(128, 255, 128, a)
#define RGBA_Lightblue(a)    ColorRGB(128, 128, 255, a)
#define RGBA_Lightcyan(a)    ColorRGB(128, 255, 255, a)
#define RGBA_Lightmagenta(a )ColorRGB(255, 128, 255, a)
#define RGBA_Lightyellow(a)  ColorRGB(255, 255, 128, a)
#define RGBA_Brightred(a)    ColorRGB(255, 192, 192, a)
#define RGBA_Brightgreen(a)  ColorRGB(192, 255, 192, a)
#define RGBA_Brightblue(a)   ColorRGB(192, 192, 255, a)
#define RGBA_Brightcyan(a)   ColorRGB(192, 255, 255, a)
#define RGBA_Brightmagenta(a)ColorRGB(255, 192, 255, a)
#define RGBA_Brightyellow(a) ColorRGB(255, 255, 192, a)
#define RGBA_Orange(a)       ColorRGB(255, 165,   0, a)

static const ColorRGB RGB_Alpha       (255, 255, 255, 192);
static const ColorRGB RGB_Translucent (255, 255, 255, 128);
static const ColorRGB RGB_Ghost       (255, 255, 255,  64);
static const ColorRGB RGB_Invisible   ( 0,    0,   0,   0);


} //namespace lpi

#endif //color_h
